// Qt includes
#include <qstring.h>
#include <qfile.h>
#include <qregexp.h>
#include <qcstring.h>

// KDE includes
#include <kstandarddirs.h>

// KMtraceViewer includes
#include "KMtraceModel.hh"
#include "KMtraceModel.moc"
#include "KMtraceLeak.hh"
#include "KMtraceCallStack.hh"
#include "KMtraceCall.hh"
#include "KMtraceSuppression.hh"

KMtraceModel::KMtraceModel( QString ktreefile ) : QObject( )
{
   // Init
   m_ktreefile = ktreefile;
   m_nofTotalAllocatedBlocks = 0;
   m_nofTotalAllocatedBytes = 0;
   m_nofMaximumAllocatedBlocks = 0;
   m_nofMaximumAllocatedBytes = 0;
   m_nofTotalLeaksBlocks = 0;
   m_nofTotalLeaksBytes = 0;
   
   readSuppressions( );
   readModel( );
}

KMtraceModel::~KMtraceModel( )
{
   writeSuppressions( );
}

// code taken from kmtrace.cc
void KMtraceModel::readExcludeFile( const char *name )
{
   int lineNo = 0;
   FILE *stream = fopen(name, "r");
   if (!stream)
   {
      fprintf(stderr, "Error: Can't open %s.\n", name);
      return;
   }
   fprintf(stderr, "Reading %s", name ); fflush( stderr );
   char line[1024];
   while(!feof(stream))
   {
      if (!fgets(line, 1023, stream)) break;
      line[strlen(line)-1] = 0;
      if ((line[0] == 0) || (line[0] == '#') || line[0] == '\n') {
         m_excludes.append( new KMtraceSuppression( line, Comment ) );
      } else {
         m_excludes.append( new KMtraceSuppression( line, MatchByString ) );
      }
      lineNo++;
      if( lineNo % 10 == 0 ) {
         fprintf( stderr, "." ); fflush( stderr );
      }
   }
   fprintf( stderr, "\n" );
   fclose(stream);
}

void KMtraceModel::writeExcludeFile( const char *name )
{
   int lineNo = 0;
   FILE *stream = fopen(name, "w");
   if (!stream)
   {
      fprintf(stderr, "Error: Can't open %s.\n", name);
      return;
   }
   fprintf(stderr, "Writting %s", name ); fflush( stderr );

   QObjectListIt iterator( m_excludes );
   QObject *obj;
   while( ( obj = iterator.current( ) ) != 0 ) {
      KMtraceSuppression *suppression = (KMtraceSuppression *)obj;
      fprintf( stream, "%s\n", suppression->getDescription( ).latin1( ) );
      lineNo++;
      if( lineNo % 10 == 0 ) {
         fprintf( stderr, "." ); fflush( stderr );
      }
      ++iterator;
   }
   fprintf( stderr, "\n" );
   fclose(stream);
}

void KMtraceModel::readSuppressions( )
{
   QCString exclude;
   
   exclude = QFile::encodeName( locateLocal( "data", "kmtrace/kde.excludes" ) );
   readExcludeFile( exclude );
}

void KMtraceModel::writeSuppressions( )
{
   QCString exclude;
   
   exclude = QFile::encodeName( locateLocal( "data", "kmtrace/kde.excludes" ) );
   writeExcludeFile( exclude );
}

void KMtraceModel::readModel( )
{
   int lineNo = 0;
   QFile file( m_ktreefile );
   fprintf(stderr, "Reading %s", m_ktreefile.latin1( ) ); fflush( stderr );

   if( file.open( IO_ReadOnly ) ) {
      QTextStream stream( &file );
      QString s;
      int line = 0;
      int pos = 0;
      QRegExp rx_stats( "^([^:]*): ([0-9]+)[^0-9]+([0-9]+)" );
      QRegExp rx_leak_header( "^\\[([0-9]+) bytes in ([0-9]+) blocks, 1st. block is ([0-9]+) bytes at ([^\\]]+)\\]" );
      QRegExp rx_leak_call( "^\\s+(\\S+)\\s+([^:]+)(:([0-9]*))?\\(([^\\+]+)(\\+([^\\)]*))?\\)$" );
      KMtraceLeak *current_leak = 0;
      
      while( !stream.eof( ) ) {
         line++;
         s = stream.readLine( );

         // the header lines
         pos = rx_stats.search( s );
         if( pos >= 0 ) {
            if( rx_stats.cap( 1 ).startsWith( "Totals allocated" ) ) {
               m_nofTotalAllocatedBlocks = rx_stats.cap( 3 ).toInt( );
               m_nofTotalAllocatedBytes = rx_stats.cap( 2 ).toInt( );
            } else if( rx_stats.cap( 1 ).startsWith( "Maximum" ) ) {
               m_nofMaximumAllocatedBlocks = rx_stats.cap( 3 ).toInt( );
               m_nofMaximumAllocatedBytes = rx_stats.cap( 2 ).toInt( );
            } else if( rx_stats.cap( 1 ).startsWith( "Totals leaked" ) ) {
               m_nofTotalLeaksBlocks = rx_stats.cap( 3 ).toInt( );
               m_nofTotalLeaksBytes = rx_stats.cap( 2 ).toInt( );
            } else if( rx_stats.cap( 1 ).startsWith( "Leaked memory after" ) ) {
               // ignore them for now because they are confusing somehow!
               int m_nofTotalLeaksFilteredBlocks = rx_stats.cap( 3 ).toInt( );
               int m_nofTotalLeaksFilteredBytes = rx_stats.cap( 2 ).toInt( );
            }
         } else {
            // the start of a leak
            pos = rx_leak_header.search( s );
            if( pos >= 0 ) {
               if( current_leak != 0 ) {
                  m_leaksList.append( current_leak );
               }
               current_leak = new KMtraceLeak( rx_leak_header.cap( 1 ).toInt( ),
                                               rx_leak_header.cap( 2 ).toInt( ),
                                               rx_leak_header.cap( 3 ).toInt( ),
                                               rx_leak_header.cap( 4 ) );
            } else {
               // call stack as part of a leak
               pos = rx_leak_call.search( s );
               if( pos >= 0 ) {
                  KMtraceCall *call = new KMtraceCall( rx_leak_call.cap( 1 ),
                                                       rx_leak_call.cap( 2 ),
                                                       rx_leak_call.cap( 4 ).toInt( ),
                                                       rx_leak_call.cap( 5 ),
                                                       rx_leak_call.cap( 7 ) );
                  current_leak->addCall( call );
               }
            }
         }
         lineNo++;
         if( lineNo % 10 == 0 ) {
            fprintf( stderr, "." ); fflush( stderr );
         }
      }
      fprintf( stderr, "\n" );

      if( current_leak != 0 ) {
         m_leaksList.append( current_leak );
      }
      
      file.close( );
   }
}

int KMtraceModel::getNofTotalAllocatedBytes( )
{
   return m_nofTotalAllocatedBytes;
}

int KMtraceModel::getNofTotalAllocatedBlocks( )
{
   return m_nofTotalAllocatedBlocks;
}

int KMtraceModel::getNofMaximumAllocatedBytes( )
{
   return m_nofMaximumAllocatedBytes;
}

int KMtraceModel::getNofMaximumAllocatedBlocks( )
{
   return m_nofMaximumAllocatedBlocks;
}

int KMtraceModel::getNofTotalLeaksBytes( )
{
   return m_nofTotalLeaksBytes;
}

int KMtraceModel::getNofTotalLeaksBlocks( )
{
   return m_nofTotalLeaksBlocks;
}

QObjectList *KMtraceModel::getLeaksList( )
{
   return &m_leaksList;
}

QObjectList *KMtraceModel::getSuppressionList( )
{
   return &m_excludes;
}

void KMtraceModel::slotSuppressionChanged( KMtraceLeak *leak )
{
   if( leak->isSuppressed( ) ) {
      m_nofTotalLeaksBlocks -= leak->getBlocks( );
      m_nofTotalLeaksBytes -= leak->getBytes( );
   } else {
      m_nofTotalLeaksBlocks += leak->getBlocks( );
      m_nofTotalLeaksBytes += leak->getBytes( );
   }
}

void KMtraceModel::addSuppression( KMtraceSuppression *suppression )
{
   m_excludes.append( suppression );
   applySuppressions( );
   emit changed( );
}

void KMtraceModel::deleteSuppression( KMtraceSuppression *suppression )
{
   m_excludes.remove( suppression );
   applySuppressions( );
   emit changed( );
}

void KMtraceModel::applySuppressions( )
{
   QObjectListIt leakIterator( m_leaksList );
   QObject *obj;
   
   while( ( obj = leakIterator.current( ) ) != 0 ) {
      KMtraceLeak *leak = (KMtraceLeak *)obj;
      leak->setSuppressed( isInSuppressList( leak ) );
      ++leakIterator;
   }
   emit changed( );
}

bool KMtraceModel::isInSuppressList( KMtraceLeak *leak )
{
   QObjectList *callStack = leak->getCallStack( );
   QObjectListIt callIterator( *callStack );
   QObject *obj2;
      
   while( ( obj2 = callIterator.current( ) ) != 0 ) {
      KMtraceCall *call = (KMtraceCall *)obj2;
      QObjectListIt suppressionIterator( m_excludes );
      QObject *obj;
      while( ( obj = suppressionIterator.current( ) ) != 0 ) {
         KMtraceSuppression *suppression = (KMtraceSuppression *)obj;
         if( suppression->getMethod( ) == MatchByString ) {
            QString function = call->getFunction( );
            QString module = call->getModule( );
            QString string = suppression->getDescription( );
            if( string == function || string == module ) {
               return true;
           }
         }
         ++suppressionIterator;
      }
      ++callIterator;
   }
   return false;
}
