#ifndef __KMTRACEMODEL__H__
#define __KMTRACEMODEL__H__

// Qt includes
#include <qobject.h>
#include <qstring.h>
#include <qobjectlist.h>

// KMtraceViewer includes
#include "KMtraceLeak.hh"
#include "KMtraceSuppression.hh"

/**
 * The model of the whole application. The model consists at the moment out
 * of the output of the kmtrace program found in the kdesdk package and
 * of the list of strings which are the suppressions which should be
 * respected. This module could be easily replaced with a module reading
 * a different format (like the one of valgrind, dmalloc, mpatrol, ...).
 *
 * @short The model of kmtraceviewer
 * @author Andreas Baumann <andreas_baumann@users.sourceforge.net>
 * @see  KMtraceLeak, KMtraceCall, KMtraceSuppression
 * @version $ProjectMajorVersion: 0 $
 */
class KMtraceModel : public QObject
{
   Q_OBJECT
   
   public:
      /**
       * Create a new model based by reading the textfile 'ktreefile'
       * and the file containing the suppressions (which is at the
       * standard place as all KDE config files).
       *
       * @param ktreefile the name of the file containing the output of
       *        the kmtrace program
       */
      KMtraceModel( QString ktreefile );

      ~KMtraceModel( );

      /**
       * Get the total number of memory blocks the program being inspected
       * allocated during its total execution time.
       *
       * @returns the total number of memory blocks allocated
       */
      int getNofTotalAllocatedBlocks( );

      /**
       * Get the total number of bytes the program being inspected
       * allocated during its total execution time.
       *
       * @returns the total number of bytes allocated
       */
      int getNofTotalAllocatedBytes( );

      /**
       * Get the maximum amount of memory blocks the program being
       * inspected consumed during its total execution time.
       *
       * @returns the maximum number of memory blocks allocated
       */
      int getNofMaximumAllocatedBlocks( );
      
      /**
       * Get the maximum amount of bytes the program being
       * inspected consumed during its total execution time.
       *
       * @returns the maximum number of memory blocks allocated
       */
      int getNofMaximumAllocatedBytes( );

      /**
       * Get the number of memory blocks the program being
       * inspected still had allocated at program termination.
       *
       * @returns the number of leaked memory blocks
       */
      int getNofTotalLeaksBlocks( );

      /**
       * Get the number of bytes the program being
       * inspected still had allocated at program termination.
       *
       * @returns the number of leaked bytes
       */
      int getNofTotalLeaksBytes( );
      
      /**
       * Get the list of leaks (each entry is of type @ref KMtraceLeak).
       *
       * @returns a list of leaks
       */
      QObjectList *getLeaksList( );

      /**
       * Get the list of suppressions (each entry is of type @ref
       * KMtraceSuppression).
       *
       * @returns a list of suppressions
       */
      QObjectList *getSuppressionList( );

      void addSuppression( KMtraceSuppression *suppression );
      void deleteSuppression( KMtraceSuppression *suppression );

      // make sure all leaks are in the correct state (respecting the
      // currently set suppression)
      void applySuppressions( );

   signals:
       /**
        * The model changed because somebody suppressed or desuppressed
        * a leak.
        */
       void changed( );

   public slots:
      void slotSuppressionChanged( KMtraceLeak *leak );
		
   private:
      // the name of the treefile
      QString m_ktreefile;

      // global statistics
      int m_nofTotalAllocatedBlocks;
      int m_nofTotalAllocatedBytes;
      int m_nofMaximumAllocatedBlocks;
      int m_nofMaximumAllocatedBytes;
      int m_nofTotalLeaksBlocks;
      int m_nofTotalLeaksBytes;

      // an object list containing all leaks
      QObjectList m_leaksList;      
      
      // a list contains all suppressed leaks
      QObjectList m_excludes;

      // read suppressed leaks from kmtrace.excludes file
      void readSuppressions( );
      void readExcludeFile( const char *name );

      // write suppressed leaks
      void writeSuppressions( );
      void writeExcludeFile( const char *name );

      // read model (this is exclude file and the current leak file)
      void readModel( );

      bool isInSuppressList( KMtraceLeak *leak );
};

#endif
