/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kdebug.h>

// Qt includes
#include <qdom.h>

// WorKflow includes
#include "conversion.h"
#include "librarydescription.h"
#include "typemanager.h"

using namespace WorKflow;

Conversion::Conversion(LibraryDescription* parent)
  : XMLDescription(), m_parent(parent)
{
  if (m_parent)
    m_parent->addConversion(this);
}

Conversion::~Conversion()
{
  unregisterConversion();
  if (m_parent)
    m_parent->removeConversion(this);
}

void Conversion::readXML(const QDomElement& e)
{
  XMLDescription::readXML(e);

  QString sourcetype = e.attribute("source");
  if (!sourcetype.isNull()) {
    m_sourceTypeId = sourcetype;
  }

  QString desttype = e.attribute("dest");
  if (!desttype.isNull()) {
    m_destTypeId = desttype;
  }

  QString action = e.attribute("action");
  if (!action.isNull()) {
    if (action == "nothing") {
      m_action = NoAction;
    } else if (action == "grep") {
      m_action = GrepAction;
    } else if (action == "script") {
      m_action = ScriptAction;
    } else if (action == "custom") {
      m_action = CustomAction;
    }
  } else {
    m_action = NoAction;
  }

  QDomNode n = e.firstChild();
  while (!n.isNull()) {
    QDomElement e = n.toElement();
    if (!e.isNull()) {
      if (e.tagName() == "expression")
        m_grepExpr = QRegExp(e.text());
    }
    n = n.nextSibling();
  }

  registerConversion();
}

void Conversion::writeXML(QDomDocument& doc, QDomElement& e)
{
  QDomElement conv = doc.createElement("conversion");

  XMLDescription::writeXML(doc, conv);

  conv.setAttribute("source", m_sourceTypeId);
  conv.setAttribute("dest", m_destTypeId);

  switch (m_action) {
    case NoAction:
      conv.setAttribute("action", "nothing");
      break;
    case GrepAction: {
      conv.setAttribute("action", "grep");
      QDomElement child = doc.createElement("expression");
      child.appendChild(doc.createTextNode(m_grepExpr.pattern()));
      conv.appendChild(child);
      break;
    }
    case ScriptAction:
      conv.setAttribute("action", "script");
      // TODO
      break;
    case CustomAction:
      conv.setAttribute("action", "custom");
      break;
  }

  e.appendChild(conv);
}

Conversion::Action Conversion::action() const
{
  return m_action;
}

Value Conversion::convert(const Value& v)
{
  switch (m_action) {
    case NoAction:
      return Value::stringList(m_destTypeId, v.asStringList());
    case GrepAction:
      return Value::stringList(m_destTypeId, v.asStringList().grep(m_grepExpr));
    case ScriptAction:
      // TODO: implement this
      return Value::stringList(m_destTypeId, v.asStringList());
    case CustomAction:
      kdWarning() << "Unimplemented conversion: " << id() << endl;
  }

  return Value::stringList(m_destTypeId, v.asStringList());
}

QString Conversion::sourceTypeId() const
{
  return m_sourceTypeId;
}

QString Conversion::destTypeId() const
{
  return m_destTypeId;
}

QRegExp Conversion::grepExpression() const
{
  return m_grepExpr;
}

void Conversion::setAction(Action action)
{
  m_action = action;
}

void Conversion::setSourceTypeId(const QString& source)
{
  m_sourceTypeId = source;
}

void Conversion::setDestTypeId(const QString& dest)
{
  m_destTypeId = dest;
}

void Conversion::setGrepExpression(const QRegExp& expr)
{
  m_grepExpr = expr;
}

void Conversion::registerConversion()
{
  TypeManager::self()->registerConversion(this);
}

void Conversion::unregisterConversion()
{
  TypeManager::self()->unregisterConversion(this);
}
