/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <ktrader.h>
#include <klibloader.h>
#include <krun.h>
#include <kplugininfo.h>
#include <kparts/componentfactory.h>
#include <kdebug.h>

// WorKflow includes
#include "librarymanager.h"
#include "library.h"

using namespace WorKflow;

class LibraryManager::Private
{
public:
  KPluginInfo* infoForId(const QString& id);

  static LibraryManager* self;

  typedef QMap<KPluginInfo*, Library*> LibraryMap;

  PluginList availableLibraries;
  LibraryList loadedLibraries;
  LibraryMap libraryMap;
};

// LibraryManager::Private implementation
KPluginInfo* LibraryManager::Private::infoForId(const QString& id)
{
  for (PluginList::ConstIterator i = availableLibraries.begin(); i != availableLibraries.end(); ++i) {
    if ((*i)->pluginName() == id)
      return *i;
  }
  return 0;
}

// LibraryManager implementation
LibraryManager* LibraryManager::Private::self = 0;

LibraryManager* LibraryManager::self()
{
  if (!Private::self) {
    Private::self = new LibraryManager();
  }
  return Private::self;
}

LibraryManager::LibraryManager()
  : QObject(0)
{
  d = new Private;
  d->availableLibraries = KPluginInfo::fromServices(KTrader::self()->query(QString::fromLatin1("WorKflow/Library")));
}

LibraryManager::~LibraryManager()
{
  delete d;
}

LibraryManager::PluginList LibraryManager::availableLibraries()
{
  return d->availableLibraries;
}

LibraryManager::LibraryList LibraryManager::loadedLibraries()
{
  return d->loadedLibraries;
}

KPluginInfo* LibraryManager::libraryInfo(Library* lib)
{
}

void LibraryManager::loadAllLibraries()
{
  // TODO: make configurable which libraries to load
  for (PluginList::ConstIterator i = d->availableLibraries.begin(); i != d->availableLibraries.end(); ++i)
    loadLibrary((*i)->pluginName());

  emit allLibrariesLoaded();
}

Library* LibraryManager::loadLibrary(const QString& id)
{
  KPluginInfo* info = d->infoForId(id);

  if (!info)
  {
    kdWarning() << k_funcinfo << "Could not find library '" << id << "'" << endl;
    return 0;
  }

  if (d->libraryMap.contains(info))
    return d->libraryMap[info];

  int error = 0;
  Library* lib = KParts::ComponentFactory::createInstanceFromQuery<Library>(QString::fromLatin1("WorKflow/Library"),
                 QString::fromLatin1("[X-KDE-PluginInfo-Name]=='%1'").arg(id), this, 0, QStringList(), &error);

  if (lib)
  {
    d->loadedLibraries.append(lib);
    d->libraryMap.insert(info, lib);

    info->setPluginEnabled(true);

//     connect( plugin, SIGNAL( destroyed( QObject * ) ), this, SLOT( slotPluginDestroyed( QObject * ) ) );
//     connect( plugin, SIGNAL( readyForUnload() ), this, SLOT( slotPluginReadyForUnload() ) );

    kdDebug() << k_funcinfo << "Successfully loaded library '" << id << "'" << endl;

    // initialize library
    lib->loadXML(info->pluginName());
    lib->init();

    emit libraryLoaded(lib);

    return lib;
  }
  else
  {
    kdWarning() << k_funcinfo << "Could not load library '" << id << "':" << endl;
    switch (error)
    {
    case KParts::ComponentFactory::ErrNoServiceFound:
      kdDebug() << "No service implementing the given mimetype and fullfilling the given constraint expression can be found." << endl;
      break;

    case KParts::ComponentFactory::ErrServiceProvidesNoLibrary:
      kdDebug() << "the specified service provides no shared library." << endl;
      break;

    case KParts::ComponentFactory::ErrNoLibrary:
      kdDebug() << "the specified library could not be loaded." << endl;
      break;

    case KParts::ComponentFactory::ErrNoFactory:
      kdDebug() << "the library does not export a factory for creating components." << endl;
      break;

    case KParts::ComponentFactory::ErrNoComponent:
      kdDebug() << "the factory does not support creating components of the specified type." << endl;
      break;
    }

    kdDebug() << "Loading plugin '" << id << "' failed, KLibLoader reported error: '" <<
            KLibLoader::self()->lastErrorMessage() << "'" << endl;
  }

  return 0;
}

void LibraryManager::unloadLibrary(const QString& id)
{
}

#include "librarymanager.moc"
