/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef SLOT_H
#define SLOT_H

// Qt includes
#include <qstring.h>
#include <qobject.h>

// WorKflow includes
#include "value.h"

// forward declarations
namespace WorKflow
{
  class Command;
  class Datatype;
  class SlotDescription;
}

namespace WorKflow
{
  /**
   * @short Base class for parameters and results in commands.
   *
   * This class is the base class for the Parameter and Result classes. Since it
   * is not meant to be instantiated directly, the constructor and destructor are
   * protected. This class mainly contains commonalities of the Parameter and
   * Result classes.
   *
   * To get the associated command, use command(); to get the slot's unique ID or
   * its name, use id() or name(), respectively. Slots (i.e. Parameters and
   * Results) are typed, i.e. only data of a certain type may be passed through a
   * slot. This type can be queried using the type() method.
   *
   * Internally, a Result can be connected to a Parameter, such that data can
   * flow between two commands.
   */
  class Slot : public QObject
  {
    Q_OBJECT

  public:
    enum Problem {
      NoProblem,
      TypeMismatch,
      MissingInput
    };

    /**
     * Returns the command instance to which this slot belongs.
     * @return The command.
     */
    Command* command();

    /**
     * Returns the slot's unique identifier.
     * @return The slot's unique ID.
     */
    QString id();

    /**
     * Returns the slot's user-visible name.
     * @return The slot's name, i18n'ed.
     */
    QString name();

    /**
     * Returns the slot's user-visible description.
     * @return The slot's description, i18n'ed.
     */
    QString description();

    /**
     * Returns the slot's current Datatype instance.
     * @return The slot's current datatype.
     */
    Datatype* type();

    /**
     * Returns the slot's possible datatype IDs.
     * @return The slot's possible datatype IDs.
     */
    QStringList typeIds();

    /**
     * Returns the slot's possible datatype IDs.
     * @return The slot's possible datatype IDs.
     */
    QString typeId();

    /**
     * Sets the slot's current datatype.
     * @param id The datatype's ID.
     */
    void setType(const QString& id);

    void resetType();

    void setProblem(Problem problem);
    Problem problem() const;

  protected:
    /**
     * Constructor.
     * @param command The command to which the slot is associated.
     * @param desc The slot's description.
     */
    Slot(Command* command, SlotDescription* desc);

    /**
     * Destructor. Disconnects the slot, if connected.
     */
    ~Slot();

    void setTypeIds(const QStringList& typeIds);

  signals:
    void guiUpdateNeeded();

  private:
    Command* m_command;
    QString m_id;
    QString m_name;
    QString m_description;
    QStringList m_typeIds;
    Datatype* m_type;
    Problem m_problem;
  };
}

#endif
