/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef TYPEMANAGER_H
#define TYPEMANAGER_H

// Qt includes
#include <qmap.h>
#include <qstring.h>
#include <qvaluelist.h>

// forward declarations
namespace WorKflow
{
  class Datatype;
  class CommandFactory;
  class Conversion;
}

namespace WorKflow
{
  /**
   * @internal
   */
  namespace Internal
  {
    struct ConversionKey
    {
      inline ConversionKey()
        : source(0), dest(0) {}
      inline ConversionKey(Datatype* s, Datatype* d)
        : source(s), dest(d) {}
      inline ConversionKey(const ConversionKey& c)
        : source(c.source), dest(c.dest) {}
      inline ConversionKey& operator=(const ConversionKey& c)
      {
        source = c.source;
        dest = c.dest;
        return *this;
      }

      Datatype* source;
      Datatype* dest;
    };

    inline bool operator<(const ConversionKey& a, const ConversionKey& b)
    {
      if (a.source == b.source)
        return a.dest < b.dest;
      else
        return a.source < b.source;
    }
  }

  /**
   * @short This class manages the available Datatype instances.
   *
   * This singleton class is used to manage the available instances of the
   * Datatype class. The singleton instance is returned by the static self()
   * method. The datatypes register and unregister themselves using
   * registerType() and unregisterType(). To query a datatype, the find() method
   * should be used.
   *
   * In addition to handling datatypes, this class also manages automatic
   * conversions between the types. Automatic conversions are preformed using
   * subclasses of the ConverisonCommand class. To register an automatic
   * conversion, use registerConversion(). To unregister a conversion, use
   * unregisterConversion(). To query for registered conversions, use
   * findConversion().
   */
  class TypeManager
  {
  public:
    typedef QValueList<Datatype*> TypeList;

    /**
     * Returns the singleton instance of this class. If the instance does not
     * exist yet, it is created.
     * @return The one and only instance of the TypeManager class.
     */
    static TypeManager* self();

    /**
     * Registers a new Datatype with the TypeManager.
     * @param type The new type to register.
     */
    void registerType(Datatype* type);

    /**
     * Removes a datatype from the list of registered types.
     * @param type The type to remove.
     */
    void unregisterType(Datatype* type);

    /**
     * Registers an automatic conversion between the datatypes @p source and
     * @p dest, using CommandFactory @p conversion.
     * @param conversion The conversion object.
     */
    void registerConversion(Conversion* conversion);

    /**
     * Removes a conversion from the list of registered automatic conversions.
     * The @p source and @p dest parameters identify the conversion uniquely.
     * @param conversion The conversion to unregister.
     */
    void unregisterConversion(Conversion* conversion);

    /**
     * Finds a datatype with the given @p id.
     * @param id The unique ID of the type to look for.
     * @return The datatype with the id @p ID, or 0 if no such type exists.
     */
    Datatype* find(const QString& id);

    TypeList types() const;

    /**
     * Searches for an automatic conversion between the @p source and @p dest
     * datatypes.
     * @param source The source datatype.
     * @param dest The destination datatype.
     * @return The object which persforms the automatic conversion between
     *   @p source and @p dest, or 0, if no such conversion exists.
     */
    Conversion* findConversion(Datatype* source, Datatype* dest);

    void rebuildConversionCache();

  private:
    typedef QValueList<Conversion*> ConversionList;

    TypeManager();
    ~TypeManager();

    Conversion* findPath(Datatype* source, Datatype* dest);
    ConversionList adj(Datatype* source);

    void addCacheEntry(Conversion* conversion);

    static TypeManager* s_self;

    typedef QMap<QString, Datatype*> TypeMap;
    typedef QMap<Internal::ConversionKey, Conversion*> ConversionMap;

    TypeMap m_typeMap;
    ConversionList m_atomarConversions;
    ConversionList m_upcastConversions;
    ConversionList m_conversionPaths;

    bool m_cacheDirty;
    ConversionMap m_conversionMap;
  };
}

#endif
