/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

#ifndef VALUELIST_H
#define VALUELIST_H

// KDE includes
#include <dcopref.h>

// Qt includes
#include <qstring.h>
#include <qvaluelist.h>
#include <qstringlist.h>
#include <qvariant.h>

// forward declarations
class QDomElement;
class QDomDocument;

namespace WorKflow
{
  class Datatype;
}

namespace WorKflow
{
  /**
   * @short Holds a WorKflow data value list.
   *
   * In WorKflow, data is processed in the form of lists. Every command processes
   * the whole list by default, so there are no "real" values, only lists. If a
   * single value is needed, a one-element list should be used.
   *
   * Lists of values are represented by this class. To get values out of the
   * list, which are internally stored as strings, use nextString(), nextNumber()
   * and nextBoolean(), respectively. These methods actually return the first
   * element of the list and remove it afterwards, so the lists are FIFOs. With
   * this approach, no complex iterator interfaces are needed.
   *
   * To fill a value list, use the addString(), addNumber() or addBoolean()
   * methods. To get the data type of the list, use type().
   */
  class Value
  {
  public:
    /**
     * Static convenience function which creates a value list containing a
     * single string.
     * @param typeId The value list's datatype's ID.
     * @param value The string value.
     */
    static Value singleString(const QString& typeId, const QString& value);

    /**
     * Static convenience function which creates a value list containing a
     * single number.
     * @param typeId The value list's datatype's ID.
     * @param value The number value.
     */
    static Value singleNumber(const QString& typeId, double value);

    /**
     * Static convenience function which creates a value list containing a
     * single boolean value.
     * @param typeId The value list's datatype's ID.
     * @param value The boolean value.
     */
    static Value singleBoolean(const QString& typeId, bool value);

    /**
     * Static convenience function which creates a value list containing a
     * list of strings.
     * @param typeId The value list's datatype's ID.
     * @param value The string list.
     */
    static Value stringList(const QString& typeId, const QStringList& list);

    /**
     * Static convenience function which creates a value list containing a
     * list of DCOP references.
     * @param typeId The value list's datatype's ID.
     * @param value The DCOPRef list.
     */
    static Value dcopRefList(const QString& typeId, const QValueList<DCOPRef>& list);

    static Value enumValue(const QString& typeId, const QString& key);
    static Value enumValue(const QString& typeId, int index);

    /**
     * Default constructor. Creates an invalid value list.
     */
    Value();

    /**
     * Constructor. Takes a pointer to the desired Datatype.
     * @param typeId The value list's datatype's ID.
     */
    explicit Value(const QString& typeId);

    /**
     * Returns if the value list is empty.
     * @return @c true if the list is empty, @c false otherwise.
     */
    bool isEmpty() const;

    /**
     * Erases all elements of the value list.
     */
    void clear();

    /**
     * Returns a string representation of the next item.
     * @return The next string item.
     */
    QString nextString();

    /**
     * Returns a numerical representation of the next item.
     * @return The next number item.
     */
    double nextNumber();

    /**
     * Returns a boolean representation of the next item.
     * @return The next boolean item.
     */
    bool nextBoolean();

    /**
     * Returns a DCOP reference representation of the next item.
     * @return The next DCOPRef item.
     */
    DCOPRef nextDcopRef();

    QString enumKey() const;
    QString enumName() const;
    int enumIndex() const;

    /**
     * Returns a list of all string in the value list.
     * @return All strings.
     */
    QStringList asStringList() const;

    QValueList<DCOPRef> asDcopRefList() const;

    /**
     * Adds a string value to the list.
     * @param value The string to add.
     */
    void addString(const QString& value);

    /**
     * Adds a number value to the list.
     * @param value The number to add.
     */
    void addNumber(double value);

    /**
     * Adds a boolean value to the list.
     * @param value The boolean value to add.
     */
    void addBoolean(bool value);

    /**
     * Adds a DCOP reference to the list.
     * @param ref The reference to add.
     */
    void addDcopRef(const DCOPRef& ref);

    /**
     * Adds a string list to the value list.
     * @param list The string list to add.
     */
    void addStringList(const QStringList& list);

    /**
     * Returns the value list's assigned Datatype.
     * @return The Datatype instance representing this list's datatype.
     */
    Datatype* type() const;

    /**
     * Converts the value list to the desired Datatype.
     * @param dest The datatype to convert the value list to.
     */
    Value convertTo(Datatype* dest);

    void readXML(const QDomElement& e);
    void writeXML(QDomDocument& doc, QDomElement& e);

    bool operator==(const Value& val) const;
    bool operator!=(const Value& val) const;

  private:
    DCOPRef stringToDcopRef(const QString& ref) const;

    Datatype* m_type;
    QStringList m_data;
  };
}

#endif
