/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <klistview.h>
#include <kpushbutton.h>
#include <dcopclient.h>
#include <kiconloader.h>
#include <kmimetype.h>

// Qt includes
#include <qlineedit.h>
// #include <qradiobutton.h>
#include <qgroupbox.h>
#include <qcombobox.h>
#include <qwidgetstack.h>

// WorKflow includes
#include "commanddescription.h"
#include "parameterdescription.h"
#include "resultdescription.h"
// #include "commandmanager.h"
// #include "category.h"

// designer includes
#include "commandruntimedialog.h"
#include "programparameterlistitem.h"
#include "dcopparameterlistitem.h"
#include "requirementlistitem.h"
#include "servicelistitem.h"
#include "dialoghelper.h"

using namespace WorKflow;

bool CommandRuntimeDialog::editCommand(QWidget* parent, CommandDescription* desc)
{
  CommandRuntimeDialog dialog(parent, desc);
  return dialog.exec();
}

// initialization
void CommandRuntimeDialog::init()
{
  initExecutionTab();
  initRequirementsTab();
  initServicesTab();
}

void CommandRuntimeDialog::initExecutionTab()
{
  m_execProgParamsListView->setSorting(-1);

  slotDcopServerChange();
}

void CommandRuntimeDialog::initRequirementsTab()
{
}

bool operator<(KMimeType::Ptr a, KMimeType::Ptr b)
{
  return (*a).name() < (*b).name();
}

void CommandRuntimeDialog::initServicesTab()
{
  KMimeType::List mimetypes = KMimeType::allMimeTypes();

  qHeapSort(mimetypes);

  for (KMimeType::List::ConstIterator i = mimetypes.begin(); i != mimetypes.end(); ++i) {
    QPixmap icon = SmallIcon((*i)->icon(QString::null, false));
    QString text = QString::fromLatin1("%1").arg((*i)->name());

    m_serviceDropMimetypeComboBox->insertItem(icon, text);
  }
}

// void CommandRuntimeDialog::updateParameterWidgets()
// {
//   ParamList list = parameters();
// 
//   QStringList strings;
// 
//   for (ParamList::ConstIterator i = list.begin(); i != list.end(); ++i)
//     strings << (*i)->id();
// 
//   m_execProgParamComboBox->clear();
//   m_execProgParamComboBox->insertStringList(strings);
//   m_execDcopParamComboBox->clear();
//   m_execDcopParamComboBox->insertStringList(strings);
//   m_serviceDropParameterComboBox->clear();
//   m_serviceDropParameterComboBox->insertStringList(strings);
// }
// 
// void CommandRuntimeDialog::updateResultWidgets()
// {
//   ResultList list = results();
// 
//   QStringList strings;
// 
//   for (ResultList::ConstIterator i = list.begin(); i != list.end(); ++i)
//     strings << (*i)->id();
// 
//   m_execDcopResultComboBox->clear();
//   m_execDcopResultComboBox->insertStringList(strings);
// }

// deinit
void CommandRuntimeDialog::accept()
{
  QDialog::accept();
}

// Constructor / Destructor
CommandRuntimeDialog::CommandRuntimeDialog(QWidget* parent, CommandDescription* desc)
  : CommandRuntimeDialogBase(parent), m_description(desc), m_dontUpdate(false)
{
  kapp->dcopClient()->setNotifications(true);

  connect(kapp->dcopClient(), SIGNAL(applicationRegistered(const QCString&)), this, SLOT(slotDcopServerChange()));
  connect(kapp->dcopClient(), SIGNAL(applicationRemoved(const QCString&)), this, SLOT(slotDcopServerChange()));

  init();
}

CommandRuntimeDialog::~CommandRuntimeDialog()
{
}

// Execution Tab

// Execute Program
void CommandRuntimeDialog::addProgramParameter()
{
  ProgramParameterListItem* item = new ProgramParameterListItem(m_execProgParamsListView);
  m_execProgParamsListView->setSelected(item, true);
}

void CommandRuntimeDialog::removeProgramParameter()
{
  DialogHelper::removeItem(m_execProgParamsListView);
}

void CommandRuntimeDialog::programParameterUp()
{
  DialogHelper::itemUp(m_execProgParamsListView);
  slotProgramParameterSelectionChanged();
}

void CommandRuntimeDialog::programParameterDown()
{
  DialogHelper::itemDown(m_execProgParamsListView);
  slotProgramParameterSelectionChanged();
}

void CommandRuntimeDialog::slotProgramParameterSelectionChanged()
{
  ProgramParameterListItem* item = static_cast<ProgramParameterListItem*>(m_execProgParamsListView->selectedItem());
  m_execProgRemoveParamButton->setEnabled(item != 0);
  m_execProgParamUpButton->setEnabled(item != 0 && item->itemAbove() != 0);
  m_execProgParamDownButton->setEnabled(item != 0 && item->itemBelow() != 0);
  m_execProgParamsEditGroupBox->setEnabled(item != 0);

  if (item) {
    m_dontUpdate = true;
    if (!DialogHelper::setComboBoxText(m_execProgParamComboBox, item->parameter()))
      item->setParameter(m_execProgParamComboBox->currentText());
    m_execProgParamSyntaxEdit->setText(item->syntax());
    m_dontUpdate = false;
  }
}

void CommandRuntimeDialog::updateProgramParameter()
{
  if (m_dontUpdate)
    return;

  ProgramParameterListItem* item = static_cast<ProgramParameterListItem*>(m_execProgParamsListView->selectedItem());
  if (!item)
    return;

  item->setParameter(m_execProgParamComboBox->currentText());
  item->setSyntax(m_execProgParamSyntaxEdit->text());
}

// Execute Dcop Call
void CommandRuntimeDialog::slotDcopServerChange()
{
  QStringList strings = DialogHelper::dcopNames();

  DialogHelper::fillEditComboBox(m_execDcopAppComboBox, strings);
  DialogHelper::fillEditComboBox(m_requirementAppComboBox, strings);
  DialogHelper::fillEditComboBox(m_serviceAppComboBox, strings);
}

void CommandRuntimeDialog::slotDcopApplicationChanged()
{
  QString appid = dcopAppId(m_execDcopAppComboBox->currentText());
  if (appid.isEmpty())
    return;

  bool ok;
  QCStringList objects = kapp->dcopClient()->remoteObjects(appid.utf8(), &ok);

  if (ok) {
    QStringList strings;
    for (QCStringList::ConstIterator i = objects.begin(); i != objects.end(); ++i) {
      QString str = *i;
      strings << str;
    }

    strings.sort();

    DialogHelper::fillEditComboBox(m_execDcopObjectComboBox, strings);
  }
}

QString CommandRuntimeDialog::dcopAppId(const QString& appname)
{
  QCStringList apps = kapp->dcopClient()->registeredApplications();

  // find first matching app id (the appname parameter is missing the -$PID suffix)
  QString appid;
  for (QCStringList::ConstIterator i = apps.begin(); i != apps.end(); ++i) {
    QString id = *i;
    if (id == appname || id.startsWith(appname + '-')) {
      appid = id;
      break;
    }
  }

  return appid;
}


void CommandRuntimeDialog::slotDcopObjectChanged()
{
  QString appid = dcopAppId(m_execDcopAppComboBox->currentText());
  if (appid.isEmpty())
    return;

  QCString obj = m_execDcopObjectComboBox->currentText().utf8();

  bool ok;
  QCStringList functions = kapp->dcopClient()->remoteFunctions(appid.utf8(), obj, &ok);

  if (ok) {
    QStringList strings;
    for (QCStringList::ConstIterator i = functions.begin(); i != functions.end(); ++i) {
      QString str = *i;
      strings << str;
    }

    strings.sort();

    DialogHelper::fillEditComboBox(m_execDcopMethodComboBox, strings);
  }
}

void CommandRuntimeDialog::slotDcopMethodChanged()
{
}

void CommandRuntimeDialog::addDcopParameter()
{
  DcopParameterListItem* item = new DcopParameterListItem(m_execDcopParamsListView);
  m_execDcopParamsListView->setSelected(item, true);
}

void CommandRuntimeDialog::removeDcopParameter()
{
  DialogHelper::removeItem(m_execDcopParamsListView);
}

void CommandRuntimeDialog::dcopParameterUp()
{
  DialogHelper::itemUp(m_execDcopParamsListView);
  slotDcopParameterSelectionChanged();
}

void CommandRuntimeDialog::dcopParameterDown()
{
  DialogHelper::itemDown(m_execDcopParamsListView);
  slotDcopParameterSelectionChanged();
}

void CommandRuntimeDialog::slotDcopParameterSelectionChanged()
{
  DcopParameterListItem* item = static_cast<DcopParameterListItem*>(m_execDcopParamsListView->selectedItem());
  m_execDcopRemoveParamButton->setEnabled(item != 0);
  m_execDcopParamUpButton->setEnabled(item != 0 && item->itemAbove() != 0);
  m_execDcopParamDownButton->setEnabled(item != 0 && item->itemBelow() != 0);
  m_execDcopParamsEditGroupBox->setEnabled(item != 0);

  if (item) {
    m_dontUpdate = true;
    if (!DialogHelper::setComboBoxText(m_execDcopParamComboBox, item->parameter()))
      item->setParameter(m_execDcopParamComboBox->currentText());
    m_dontUpdate = false;
  }
}

void CommandRuntimeDialog::updateDcopParameter()
{
  if (m_dontUpdate)
    return;

  DcopParameterListItem* item = static_cast<DcopParameterListItem*>(m_execDcopParamsListView->selectedItem());
  if (!item)
    return;

  item->setParameter(m_execDcopParamComboBox->currentText());
}

// Requirements Tab
void CommandRuntimeDialog::addRequirement()
{
  RequirementListItem* item = new RequirementListItem(m_requirementsListView);
  m_requirementsListView->setSelected(item, true);
}

void CommandRuntimeDialog::removeRequirement()
{
  DialogHelper::removeItem(m_requirementsListView);
}

void CommandRuntimeDialog::slotRequirementSelectionChanged()
{
  RequirementListItem* item = static_cast<RequirementListItem*>(m_requirementsListView->selectedItem());
  m_removeRequirementButton->setEnabled(item != 0);
  m_editRequirementGroupBox->setEnabled(item != 0);

  if (item) {
    m_dontUpdate = true;
    m_requirementKindComboBox->setCurrentItem(item->kind());
    m_requirementKindWidgetStack->raiseWidget(item->kind());
    m_requirementProgramEdit->setText(item->application());
    m_requirementAppComboBox->setEditText(item->application());
    m_dontUpdate = false;
  }
}

void CommandRuntimeDialog::updateRequirement()
{
  if (m_dontUpdate)
    return;

  RequirementListItem* item = static_cast<RequirementListItem*>(m_requirementsListView->selectedItem());
  if (item) {
    item->setKind(RequirementListItem::Kind(m_requirementKindComboBox->currentItem()));
    if (item->kind() == RequirementListItem::ProgramInstalled)
      item->setApplication(m_requirementProgramEdit->text());
    else
      item->setApplication(m_requirementAppComboBox->currentText());
  }
}

// Services Tab
void CommandRuntimeDialog::addService()
{
  ServiceListItem* item = new ServiceListItem(m_servicesListView);
  m_servicesListView->setSelected(item, true);
}

void CommandRuntimeDialog::removeService()
{
  DialogHelper::removeItem(m_servicesListView);
}

void CommandRuntimeDialog::slotServiceSelectionChanged()
{
  ServiceListItem* item = static_cast<ServiceListItem*>(m_servicesListView->selectedItem());

  m_removeServiceButton->setEnabled(item != 0);
  m_editServiceGroupBox->setEnabled(item != 0);

  if (item) {
    m_dontUpdate = true;
    m_serviceKindComboBox->setCurrentItem(item->kind());
    m_serviceKindWidgetStack->raiseWidget(item->kind());

    m_serviceDropMimetypeComboBox->setEditText(item->mimeType());
    if (!DialogHelper::setComboBoxText(m_serviceDropParameterComboBox, item->parameter()))
      item->setParameter(m_serviceDropParameterComboBox->currentText());
    m_serviceAppComboBox->setEditText(item->application());
    if (!DialogHelper::setComboBoxText(m_serviceModifyHelperComboBox, item->command()))
      item->setCommand(m_serviceModifyHelperComboBox->currentText());
    m_serviceModifyWarningEdit->setText(item->message());
    m_serviceModifyAddEdit->setText(item->addButton());
    m_serviceModifyDontAddEdit->setText(item->dontAddButton());
    m_dontUpdate = false;
  }
}

void CommandRuntimeDialog::updateService()
{
  if (m_dontUpdate)
    return;

  ServiceListItem* item = static_cast<ServiceListItem*>(m_servicesListView->selectedItem());
  if (item) {
    item->setKind(ServiceListItem::Kind(m_serviceKindComboBox->currentItem()));

    item->setMimeType(m_serviceDropMimetypeComboBox->currentText());
    item->setParameter(m_serviceDropParameterComboBox->currentText());
    item->setApplication(m_serviceAppComboBox->currentText());
    item->setCommand(m_serviceModifyHelperComboBox->currentText());
    item->setMessage(m_serviceModifyWarningEdit->text());
    item->setAddButton(m_serviceModifyAddEdit->text());
    item->setDontAddButton(m_serviceModifyDontAddEdit->text());
  }
}

#include "commandruntimedialog.moc"
