/***************************************************************************
 *   Copyright (C) 2006 by Thomas Kadauke                                  *
 *   tkadauke@gmx.de                                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,      *
 *   Boston, MA 02110-1301, USA.                                           *
 ***************************************************************************/

// KDE includes
#include <kpushbutton.h>
#include <kinputdialog.h>
#include <klocale.h>

// Qt includes
#include <qlistbox.h>
#include <qlayout.h>

// WorKflow includes
#include "parameter.h"

// widget includes
#include "stringlisteditor.h"

using namespace WorKflow;

class StringListEditor::Private
{
public:
  QListBox* listBox;
  KPushButton* removeButton;
  KPushButton* upButton;
  KPushButton* downButton;
};

StringListEditor::StringListEditor(QWidget* parent, const char* name, WFlags fl)
  : QWidget(parent, name, fl), WidgetInterface()
{
  d = new Private;

  QGridLayout* layout = new QGridLayout(this, 4, 2);
  d->listBox = new QListBox(this);
  layout->addMultiCellWidget(d->listBox, 0, 3, 1, 1);

  KPushButton* addButton = new KPushButton(KStdGuiItem::guiItem(KStdGuiItem::Add), this);
  layout->addWidget(addButton, 0, 1);

  d->removeButton = new KPushButton(KStdGuiItem::guiItem(KStdGuiItem::Remove), this);
  layout->addWidget(d->removeButton, 1, 1);

  d->upButton = new KPushButton(i18n("Up"), this);
  layout->addWidget(d->upButton, 2, 1);

  d->downButton = new KPushButton(i18n("Down"), this);
  layout->addWidget(d->downButton, 3, 1);

  connect(addButton, SIGNAL(clicked()), this, SLOT(add()));
  connect(d->removeButton, SIGNAL(clicked()), this, SLOT(remove()));
  connect(d->upButton, SIGNAL(clicked()), this, SLOT(up()));
  connect(d->downButton, SIGNAL(clicked()), this, SLOT(down()));

  connect(d->listBox, SIGNAL(selectionChanged()), this, SLOT(slotSelectionChanged()));

  slotSelectionChanged();
}

StringListEditor::~StringListEditor()
{
  delete d;
}

void StringListEditor::add()
{
  bool ok;
  QString item = KInputDialog::getText(i18n("Add Item"), i18n("Please enter a string"), "", &ok, this);
  if (ok) {
    d->listBox->insertItem(item);
    d->listBox->setCurrentItem(d->listBox->count() - 1);
  }

  emitChanged();
}

void StringListEditor::remove()
{
  if (d->listBox->currentItem() == -1)
    return;

  d->listBox->removeItem(d->listBox->currentItem());

  emitChanged();
}

void StringListEditor::up()
{
  if (d->listBox->currentItem() == -1)
    return;

  QString item = d->listBox->currentText();
  int index = d->listBox->currentItem();
  if (index == 0)
    return;

  d->listBox->removeItem(index);
  d->listBox->insertItem(item, index - 1);
  d->listBox->setCurrentItem(index - 1);

  emitChanged();
}

void StringListEditor::down()
{
  if (d->listBox->currentItem() == -1)
    return;

  QString item = d->listBox->currentText();
  int index = d->listBox->currentItem();
  if (index >= d->listBox->count() - 1)
    return;

  d->listBox->removeItem(index);
  d->listBox->insertItem(item, index + 1);
  d->listBox->setCurrentItem(index + 1);

  emitChanged();
}

void StringListEditor::slotSelectionChanged()
{
  int current = d->listBox->currentItem();
  d->removeButton->setEnabled(current != -1);
  d->upButton->setEnabled(current > 0);
  d->downButton->setEnabled(current < d->listBox->count() - 1);
}

void StringListEditor::emitChanged()
{
  if (!parameter())
    return;

  emit changed(value());
}

Value StringListEditor::value()
{
  QStringList list;
  for (unsigned int i = 0; i != d->listBox->count(); ++i)
    list << d->listBox->text(i);

  return Value::stringList(parameter()->typeId(), list);
}

void StringListEditor::setParameterId(const QString& paramId)
{
  WidgetInterface::setParameterId(paramId);
}

QString StringListEditor::parameterId() const
{
  return WidgetInterface::parameterId();
}

void StringListEditor::setValue(const Value& value)
{
  QStringList strings = value.asStringList();

  d->listBox->clear();
  d->listBox->insertStringList(strings);
}

#include "stringlisteditor.moc"
