/*
 * Copyright (C) 2018 Eike Hein <hein@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
*/

import QtQuick 2.5
import QtQuick.Controls 2.5 as QQC2
import QtQuick.Layouts 1.1

import org.kde.kcm 1.2
import org.kde.kirigami 2.10 as Kirigami
import org.kde.plasma.core 2.1 as PlasmaCore

ScrollViewKCM {
    id: root

    ConfigModule.quickHelp: i18n("This module lets you configure the navigation, number and layout of virtual desktops.")

    Connections {
        target: kcm.desktopsModel

        onReadyChanged: {
            rowsSpinBox.value = kcm.desktopsModel.rows;
        }

        onRowsChanged: {
            rowsSpinBox.value = kcm.desktopsModel.rows;
        }
    }

    Component {
        id: desktopsListItemComponent

        Kirigami.SwipeListItem {
            id: listItem

            contentItem: RowLayout {
                QQC2.TextField {
                    id: nameField

                    background: null
                    leftPadding: Kirigami.Units.largeSpacing
                    topPadding: 0
                    bottomPadding: 0

                    Layout.fillWidth: true

                    Layout.alignment: Qt.AlignVCenter

                    text: model.display

                    readOnly: true

                    onEditingFinished: {
                        readOnly = true;
                        Qt.callLater(kcm.desktopsModel.setDesktopName, model.Id, text);
                    }
                }
            }

        actions: [
            Kirigami.Action {
                enabled: !model.IsMissing
                iconName: "edit-rename"
                tooltip: i18nc("@info:tooltip", "Rename")
                onTriggered: {
                    nameField.readOnly = false;
                    nameField.selectAll();
                    nameField.forceActiveFocus();
                }
            },
            Kirigami.Action {
                enabled: !model.IsMissing
                iconName: "edit-delete-remove"
                tooltip: i18nc("@info:tooltip", "Remove")
                onTriggered: kcm.desktopsModel.removeDesktop(model.Id)
            }]
        }
    }

    header: ColumnLayout {
        id: messagesLayout

        spacing: Kirigami.Units.largeSpacing

        Kirigami.InlineMessage {
            Layout.fillWidth: true

            type: Kirigami.MessageType.Error

            text: kcm.desktopsModel.error

            visible: kcm.desktopsModel.error != ""
        }

        Kirigami.InlineMessage {
            Layout.fillWidth: true

            type: Kirigami.MessageType.Information

            text: i18n("Virtual desktops have been changed outside this settings application. Saving now will overwrite the changes.")

            visible: kcm.desktopsModel.serverModified
        }
    }

    view: ListView {
        id: desktopsList

        model: kcm.desktopsModel.ready ? kcm.desktopsModel : null

        section.property: "DesktopRow"
        section.delegate: Kirigami.ListSectionHeader {
            width: desktopsList.width
            label: i18n("Row %1", section)
        }

        delegate: Kirigami.DelegateRecycler {
            width: desktopsList.width

            sourceComponent: desktopsListItemComponent
        }
    }

    footer: ColumnLayout {
        RowLayout {
            QQC2.Button {
                text: i18nc("@action:button", "Add")
                icon.name: "list-add"

                onClicked: kcm.desktopsModel.createDesktop(i18n("New Desktop"))
            }

            Item { // Spacer
                Layout.fillWidth: true
            }

            QQC2.SpinBox {
                id: rowsSpinBox

                from: 1
                to: 20
                editable: true

                textFromValue: function(value, locale) { return i18np("1 Row", "%1 Rows", value)}

                onValueModified: kcm.desktopsModel.rows = value
            }
        }

        Kirigami.FormLayout {
            Connections {
                target: kcm

                onNavWrapsChanged: navWraps.checked = kcm.navWraps

                onOsdEnabledChanged: osdEnabled.checked = kcm.osdEnabled
                onOsdDurationChanged: osdDuration.value = kcm.osdDuration
                onOsdTextOnlyChanged: osdTextOnly.checked = !kcm.osdTextOnly
            }

            QQC2.CheckBox {
                id: navWraps

                Kirigami.FormData.label: i18n("Options:")

                text: i18n("Navigation wraps around")

                checked: kcm.navWraps

                onCheckedChanged: kcm.navWraps = checked
            }

            RowLayout {
                Layout.fillWidth: true

                QQC2.CheckBox {
                    id: animationEnabled

                    text: i18n("Show animation when switching:")

                    checked: kcm.animationsModel.enabled

                    onCheckedChanged: kcm.animationsModel.enabled = checked
                }

                QQC2.ComboBox {
                    enabled: animationEnabled.checked

                    model: kcm.animationsModel
                    textRole: "NameRole"
                    currentIndex: kcm.animationsModel.currentIndex
                    onActivated: kcm.animationsModel.currentIndex = currentIndex
                }

                QQC2.Button {
                    enabled: animationEnabled.checked && kcm.animationsModel.currentConfigurable

                    icon.name: "configure"

                    onClicked: kcm.configureAnimation()
                }

                QQC2.Button {
                    enabled: animationEnabled.checked

                    icon.name: "dialog-information"

                    onClicked: kcm.showAboutAnimation()
                }

                Item {
                    Layout.fillWidth: true
                }
            }

            RowLayout {
                Layout.fillWidth: true

                QQC2.CheckBox {
                    id: osdEnabled

                    text: i18n("Show on-screen display when switching:")

                    checked: kcm.osdEnabled

                    onToggled: kcm.osdEnabled = checked
                }

                QQC2.SpinBox {
                    id: osdDuration

                    enabled: osdEnabled.checked

                    from: 0
                    to: 10000
                    stepSize: 100

                    textFromValue: function(value, locale) { return i18n("%1 ms", value)}

                    value: kcm.osdDuration

                    onValueChanged: kcm.osdDuration = value
                }
            }

            RowLayout {
                Layout.fillWidth: true

                Item {
                    width: units.largeSpacing
                }

                QQC2.CheckBox {
                    id: osdTextOnly

                    enabled: osdEnabled.checked

                    text: i18n("Show desktop layout indicators")

                    checked: !kcm.osdTextOnly

                    onToggled: kcm.osdTextOnly = !checked
                }
            }
        }
    }
}

