/*
    This file is part of KOrganizer.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include <qtooltip.h>
#include <qfiledialog.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qbuttongroup.h>
#include <qvgroupbox.h>
#include <qwidgetstack.h>
#include <qdatetime.h>
#include <qgrid.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qtextcodec.h>

#include <kdebug.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kdebug.h>
#ifndef KORG_NOKABC
#include <kabc/addresseedialog.h>
#include <kabc/stdaddressbook.h>
#endif

#include <libkcal/incidence.h>
#include <libkcal/freebusy.h>

#include "koprefs.h"
#include "kogroupware.h"
#include "koeventtimewidget.h"
#include "kolineedit.h"

#include "koeditordetails.h"
#include "koeditordetails.moc"

#include "kdgantt/KDGanttViewTaskItem.h"

#include <assert.h>


// We can't use the CustomListViewItem base class, since we need a
// different inheritance hierarchy for supporting the Gantt view.
class AttendeeListItem : public KDGanttViewTaskItem
{
  public:
    AttendeeListItem( Attendee* data, KDGanttView *parent ) :
      KDGanttViewTaskItem( parent ), mData( data ), mFreeBusy( 0 )
    {
      assert( data );
      updateItem();
    }
    ~AttendeeListItem();

    void updateItem();

    Attendee* data() const { return mData; }
    void setFreeBusy( KCal::FreeBusy* fb ) { mFreeBusy = fb; }
    KCal::FreeBusy* freeBusy() const { return mFreeBusy; }

    void setFreeBusyPeriods( FreeBusy* fb );

    QString key(int column, bool) const
    {
      QMap<int,QString>::ConstIterator it = mKeyMap.find(column);
      if (it == mKeyMap.end()) return listViewText(column);
      else return *it;
    }

    void setSortKey(int column,const QString &key)
    {
      mKeyMap.insert(column,key);
    }

    QString email() const { return mData->email(); }

  private:
    Attendee* mData;
    KCal::FreeBusy* mFreeBusy;

    QMap<int,QString> mKeyMap;
};



AttendeeListItem::~AttendeeListItem()
{
  delete mData;
}

void AttendeeListItem::updateItem()
{
  setListViewText(0,mData->name());
  setListViewText(1,mData->email());
  setListViewText(2,mData->roleStr());
  setListViewText(3,mData->statusStr());
  if (mData->RSVP() && !mData->email().isEmpty())
    setPixmap(4,SmallIcon("mailappt"));
  else
    setPixmap(4,SmallIcon("nomailappt"));
}


// Set the free/busy periods for this attendee
void AttendeeListItem::setFreeBusyPeriods( FreeBusy* fb )
{
  if( fb ) {
    // Clean out the old entries
    for( KDGanttViewItem* it = firstChild(); it; it = firstChild() )
      delete it;

    // Evaluate free/busy information
    QValueList<KCal::Period> busyPeriods = fb->busyPeriods();
    for( QValueList<KCal::Period>::Iterator it = busyPeriods.begin();
	 it != busyPeriods.end(); ++it ) {
      KDGanttViewTaskItem* newSubItem = new KDGanttViewTaskItem( this );
      newSubItem->setStartTime( (*it).start() );
      newSubItem->setEndTime( (*it).end() );
      newSubItem->setColors( Qt::red, Qt::red, Qt::red );
    }
    setFreeBusy( fb );
    setShowNoInformation( false );
  } else {
    // No free/busy information
    setFreeBusy( 0 );
    setShowNoInformation( true );
  }
}


KOEditorDetails::KOEditorDetails (bool showDateTime, int spacing,QWidget* parent,const char* name)
  : QWidget( parent, name), mDisableItemUpdate( false )
{
  QVBoxLayout *topLayout = new QVBoxLayout(this);
  topLayout->setSpacing(spacing);

  if( !KOPrefs::instance()->mGroupwareCommunication ) {
    QLabel *l = new QLabel( i18n( "This is disabled when you're not running korganizer in KMail" ), this );
    l->setAlignment( AlignCenter );
    topLayout->addWidget( l );
    return;
  }

  QString organizer = KOPrefs::instance()->email();
  mOrganizerLabel = new QLabel(i18n("Organizer: %1").arg(organizer),this);
  topLayout->addWidget( mOrganizerLabel );
  mIsOrganizer = KOPrefs::instance()->email() == organizer;

  mEventTimeWidget = new KOEventTimeWidget( this );
  topLayout->addWidget( mEventTimeWidget );
  if( !showDateTime )
      mEventTimeWidget->hide();

  QVGroupBox* participantsVGB = new QVGroupBox( i18n( "Participants" ), this );
  participantsVGB->setMinimumHeight( 400 );
  topLayout->addWidget( participantsVGB );

  // Label for status summary information:
  mStatusSummaryLabel = new QLabel( participantsVGB );
  // Use the tooltip palette for this info 
  // widget to highlight it
  mStatusSummaryLabel->setPalette( QToolTip::palette() );
  mStatusSummaryLabel->setFrameStyle( QFrame::Plain | QFrame::Box );
  mStatusSummaryLabel->setLineWidth( 1 );
  if( !mIsOrganizer )
    mStatusSummaryLabel->hide();

  // scale HB also has the Pick button
  QHBox* scaleHB = new QHBox( participantsVGB );
  QLabel* scaleLA = new QLabel( i18n( "Scale: " ), scaleHB );
  scaleLA->setAlignment( AlignRight | AlignVCenter );
  scaleCombo = new QComboBox( scaleHB ); 
  QLabel* dummy = new QLabel( scaleHB );
  scaleHB->setStretchFactor( dummy, 2 );
  QLabel* hFormatLA = new QLabel( i18n( "Hour Format: " ), scaleHB );
  hFormatLA->setAlignment( AlignRight | AlignVCenter );
  QComboBox* hFormatCombo = new QComboBox( scaleHB );
  dummy = new QLabel( scaleHB );
  scaleHB->setStretchFactor( dummy, 2 );
  QPushButton* centerPB = new QPushButton( i18n( "Center on Start" ), scaleHB );
  connect( centerPB, SIGNAL( clicked() ), this, SLOT( slotCenterOnStart() ) );
  dummy = new QLabel( scaleHB );
  scaleHB->setStretchFactor( dummy, 2 );
  QPushButton* zoomPB = new QPushButton( i18n( "Zoom to fit" ), scaleHB );
  connect( zoomPB, SIGNAL( clicked() ), this, SLOT( slotZoomToTime() ) );
  dummy = new QLabel( scaleHB );
  scaleHB->setStretchFactor( dummy, 2 );
  QPushButton* pickPB = new QPushButton( i18n( "Pick a date" ), scaleHB );
  connect( pickPB, SIGNAL( clicked() ), this, SLOT( slotPickDate() ) );

  scaleCombo->insertItem( i18n( "Hour" ) );
  scaleCombo->insertItem( i18n( "Day" ) );
  scaleCombo->insertItem( i18n( "Week" ) );
  scaleCombo->insertItem( i18n( "Month" ) );
  scaleCombo->insertItem( i18n( "Automatic" ) );
  scaleCombo->setCurrentItem( 0 ); // start with "hour"

  
  connect( scaleCombo, SIGNAL( activated( int ) ),
           this, SLOT( slotScaleChanged( int ) ) );
  connect( hFormatCombo, SIGNAL( activated( int ) ),
           this, SLOT( slothFormatChanged( int ) ) );

  hFormatCombo->insertItem( i18n( "24:00 View" ) );
  hFormatCombo->insertItem( i18n( "12:PM View" ) );
  hFormatCombo->setCurrentItem( 1 );
  mGanttView = new KDGanttView(participantsVGB,"mGanttView");
  // Remove the predefined "Task Name" column
  mGanttView->removeColumn( 0 );
  mGanttView->addColumn(i18n("Name"),180);
  mGanttView->addColumn(i18n("Email"),180);
  mGanttView->addColumn(i18n("Role"),60);
  mGanttView->addColumn(i18n("Status"),100);
  mGanttView->addColumn(i18n("RSVP"),35);
  if ( KOPrefs::instance()->mCompactDialogs ) {
    mGanttView->setFixedHeight(78);
  }
  mGanttView->setHeaderVisible( true );
  mGanttView->setScale( KDGanttView::Hour );
  if ( !strcmp(QTextCodec::codecForLocale()->locale(),"de_DE@euro") ) {
    mGanttView->setHourFormat( KDGanttView::Hour_24   );
    hFormatCombo->setCurrentItem( 0 );
  }
  // Initially, show 15 days back and forth
  // set start to even hours, i.e. to 12:AM 0 Min 0 Sec
  QDateTime horizonStart = QDateTime( QDateTime::currentDateTime().addDays( -15 ).date() );
  QDateTime horizonEnd = QDateTime::currentDateTime().addDays( 15 );
  mGanttView->setHorizonStart( horizonStart );
  mGanttView->setHorizonEnd( horizonEnd );
  mGanttView->setCalendarMode( true );
  mGanttView->setDisplaySubitemsAsGroup( true );
  mGanttView->setShowLegendButton( false );
  //mGanttView->setFixedHeight( 400 );
  // Initially, center to current date
  //mGanttView->show();
  mGanttView->centerTimelineAfterShow( QDateTime::currentDateTime() );

  connect(mGanttView,SIGNAL(lvSelectionChanged(KDGanttViewItem *)),
          SLOT(updateAttendeeInput()));
  connect(mGanttView,SIGNAL(lvItemDoubleClicked(KDGanttViewItem *)),
          SLOT(updateFreeBusyData()));

  QHBox* lowerHB = new QHBox( participantsVGB );
  lowerHB->setSpacing( spacing );
  QGrid* personGrid = new QGrid( 2, lowerHB );
  personGrid->setSpacing( spacing );

  QLabel *attendeeLabel = new QLabel(personGrid);
  attendeeLabel->setText(i18n("Name:"));

  mNameEdit = new KOLineEdit(this, true, personGrid, "KOEditorDetails::mNameEdit" );
  connect(mNameEdit,SIGNAL(textChanged(const QString &)),
          SLOT(updateAttendeeItem()));

  mUidEdit = new QLineEdit(0);
  mUidEdit->setText("");
  mUidEdit->hide(); // for now

  QLabel *emailLabel = new QLabel(personGrid);
  emailLabel->setText(i18n("Email:"));

  mEmailEdit = new KOFocusOutLineEdit(personGrid);
  connect(mEmailEdit,SIGNAL(textChanged(const QString &)),
          SLOT(updateAttendeeItem()));
  connect(mEmailEdit,SIGNAL(focusLost()),
          SLOT(updateFreeBusyData()));
  mNameEdit->setEmailLineEdit( mEmailEdit );

  QLabel *attendeeRoleLabel = new QLabel(personGrid);
  attendeeRoleLabel->setText(i18n("Role:"));

  mRoleCombo = new QComboBox(false,personGrid);
  mRoleCombo->insertStringList(Attendee::roleList());
  connect(mRoleCombo,SIGNAL(activated(int)),SLOT(updateAttendeeItem()));

  QLabel *statusLabel = new QLabel(personGrid);
  statusLabel->setText( i18n("Status:") );

  mStatusCombo = new QComboBox(false,personGrid);
  mStatusCombo->insertStringList(Attendee::statusList());
  connect(mStatusCombo,SIGNAL(activated(int)),SLOT(updateAttendeeItem()));

  mRsvpButton = new QCheckBox(participantsVGB);
  mRsvpButton->setText(i18n("Request response"));
  connect(mRsvpButton,SIGNAL(clicked()),SLOT(updateAttendeeItem()));

  QVBox* buttonBox = new QVBox( lowerHB );
  buttonBox->setSpacing( spacing );

  QPushButton *newButton = new QPushButton(i18n("&New"),buttonBox);
  connect(newButton,SIGNAL(clicked()),SLOT(addNewAttendee()));

  mRemoveButton = new QPushButton(i18n("&Remove"),buttonBox);
  connect(mRemoveButton, SIGNAL(clicked()),SLOT(removeAttendee()));

  mAddressBookButton = new QPushButton(i18n("Address &Book..."),buttonBox);
  connect(mAddressBookButton,SIGNAL(clicked()),SLOT(openAddressBook()));

  // Add the organizer himself, if there is nobody else - Outlook does
  // it this way as well.
  /*
  if( !mGanttView->childCount() ) {
      Attendee *a = new Attendee( KOPrefs::instance()->fullName(),  KOPrefs::instance()->email() );
      insertAttendee(a);
  }
  */
#ifdef KORG_NOKABC
  mAddressBookButton->hide();
#endif

  updateAttendeeInput();
}

KOEditorDetails::~KOEditorDetails()
{
}

void KOEditorDetails::insertMyself()
{
  if( KOPrefs::instance()->mGroupwareCommunication &&
      mGanttView && mGanttView->childCount() == 0 )
    insertAttendee( new Attendee( KOPrefs::instance()->fullName(),
				  KOPrefs::instance()->email(),
				  false, Attendee::Accepted ) );
  updateStatusSummary();
}


void KOEditorDetails::slotScaleChanged( int newScale )
{
    // The +1 is for the Minute scale which we don't offer in the
    // combo box.
    KDGanttView::Scale scale = static_cast<KDGanttView::Scale>( newScale+1 );
    mGanttView->setScale( scale );
    mGanttView->centerTimeline(mEventTimeWidget->startDateTime());
}

void KOEditorDetails::removeAttendee()
{
  AttendeeListItem *aItem = (AttendeeListItem *)mGanttView->selectedItem();
  if (!aItem) return;

  Attendee *delA = new Attendee(aItem->data()->name(),aItem->data()->email(),
    aItem->data()->RSVP(),aItem->data()->status(),aItem->data()->role(),
    aItem->data()->uid());
  mdelAttendees.append(delA);

  delete aItem;

  updateAttendeeInput();
  updateStatusSummary();
}


void KOEditorDetails::openAddressBook()
{
#ifndef KORG_NOKABC
  // Refresh the addressbook in case someone added new entries
  KABC::StdAddressBook::self()->load();

  // ... and run the dialog
  KABC::Addressee attendee = KABC::AddresseeDialog::getAddressee(this);
  if (!attendee.isEmpty()) {
    bool found = false;
    QListViewItem* item;
    for (item = mGanttView->firstChild(); item && !found; item = item->nextSibling()) {
      Attendee* a = static_cast<AttendeeListItem*>(item)->data();
      if( a->name() == attendee.realName() && a->email() == attendee.preferredEmail() )
	found = true;
    }

    if( !found || KMessageBox::Yes ==
	KMessageBox::questionYesNo(this,
				   i18n("Do you really want to add the same person again?.")))
      insertAttendee( new Attendee( attendee.realName(), attendee.preferredEmail(), true,
				    KCal::Attendee::NeedsAction,
				    KCal::Attendee::ReqParticipant, attendee.uid() ) );
  }
#endif
}


void KOEditorDetails::addNewAttendee()
{
  insertAttendee( new Attendee( i18n("(EmptyName)"), i18n("(EmptyEmail)"), true ) );
}


void KOEditorDetails::insertAttendee(Attendee *a)
{
  AttendeeListItem *item = new AttendeeListItem(a,mGanttView);
  mGanttView->setSelected( item, true );
  if( a->name() != "(EmptyName)") updateFreeBusyData();
  updateStatusSummary();
}

void KOEditorDetails::setDefaults( QDateTime dtStart, QDateTime dtEnd,
                                   bool allDay )
{
  if( KOPrefs::instance()->mGroupwareCommunication ) {
    mEventTimeWidget->setDefaults( dtStart, dtEnd, allDay );
    slotUpdateGanttView( dtStart, dtEnd );
  }
}

void KOEditorDetails::readEvent(Incidence *incidence, bool tmpl )
{
  if( !KOPrefs::instance()->mGroupwareCommunication )
    return;

  // Transfer the time data if the incidence is an event. This is
  // bad style, but saves us from creating a bunch of subclasses of
  // KOEditorDetails.
  KCal::Event* event = 0;
  if( incidence->type() == "Event" )
    event = static_cast<KCal::Event*>( incidence );
  if( event )
    mEventTimeWidget->readEvent( event, tmpl );

  // Clear out the Gantt View again so that we get rid of the
  // default attendee. The way KOEditorDetails is setup, this is
  // unfortunately necessary :-(
  bool block = mGanttView->getUpdateEnabled( );
  mGanttView->setUpdateEnabled( false );
  mGanttView->clear();

  mdelAttendees.clear();
  QPtrList<Attendee> tmpAList = incidence->attendees();
  Attendee *a;
  for (a = tmpAList.first(); a; a = tmpAList.next())
    insertAttendee(new Attendee(*a));
  mGanttView->setSelected( mGanttView->firstChild(), true );
  mOrganizerLabel->setText(i18n("Organizer: %1").arg(incidence->organizer()));
  mIsOrganizer = KOPrefs::instance()->email() == incidence->organizer();
  if( event ) {
    mGanttView->centerTimelineAfterShow( event->dtStart());
    mGanttView->clearBackgroundColor();
    mGanttView->setIntervalBackgroundColor( event->dtStart(), event->dtEnd(), Qt::magenta );
  } 
  mGanttView->setUpdateEnabled( block );
  updateStatusSummary();
}
void KOEditorDetails::slotCenterOnStart() 
{
  mGanttView->centerTimeline(mEventTimeWidget->startDateTime());
}
void KOEditorDetails::slotZoomToTime() 
{
  bool block  = mGanttView->getUpdateEnabled();
  mGanttView->setUpdateEnabled( false );
  if ( scaleCombo->currentItem() != 4 ) {
    scaleCombo->setCurrentItem( 4 );// auto
    slotScaleChanged( 4 );// auto
  }
  mGanttView->setUpdateEnabled( block );
  mGanttView->zoomToSelection(mEventTimeWidget->startDateTime() ,mEventTimeWidget->endDateTime());
}
void KOEditorDetails::slothFormatChanged( int i ) 
{
  if ( i )
    mGanttView->setHourFormat( KDGanttView::Hour_12 );
  else
    mGanttView->setHourFormat( KDGanttView::Hour_24 );
}
void KOEditorDetails::writeEvent(Incidence *event)
{
  if( !KOPrefs::instance()->mGroupwareCommunication )
    return;

  event->clearAttendees();
  QListViewItem *item;
  AttendeeListItem *a;
  for (item = mGanttView->firstChild(); item;
       item = item->nextSibling()) {
    a = (AttendeeListItem *)item;
    event->addAttendee(new Attendee(*(a->data())));
  }
}

void KOEditorDetails::cancelAttendeeEvent(Incidence *event)
{
  event->clearAttendees();
  Attendee * att;
  for (att=mdelAttendees.first();att;att=mdelAttendees.next()) {
    event->addAttendee(new Attendee(*att));
  }
  mdelAttendees.clear();
}

bool KOEditorDetails::validateInput()
{
  return true;
}

void KOEditorDetails::updateAttendeeInput()
{
  QListViewItem *item = mGanttView->selectedItem();
  AttendeeListItem *aItem = static_cast<AttendeeListItem *>( item );
  if (aItem) {
    fillAttendeeInput( aItem );
    mNameEdit->setFocus();
    mNameEdit->selectAll();
  } else {
    clearAttendeeInput();
  }
}

void KOEditorDetails::clearAttendeeInput()
{
  mNameEdit->setText("");
  mUidEdit->setText("");
  mEmailEdit->setText("");
  mRoleCombo->setCurrentItem(0);
  mStatusCombo->setCurrentItem(0);
  mRsvpButton->setChecked(true);
  setEnabledAttendeeInput( false );
}

void KOEditorDetails::fillAttendeeInput( AttendeeListItem *aItem )
{
  Attendee *a = aItem->data();
  mDisableItemUpdate = true;
  mNameEdit->setText(a->name());
  mUidEdit->setText(a->uid());
  mEmailEdit->setText(a->email());
  mRoleCombo->setCurrentItem(a->role());
  mStatusCombo->setCurrentItem(a->status());
  mRsvpButton->setChecked(a->RSVP());

  mDisableItemUpdate = false;

  setEnabledAttendeeInput( true );
}

void KOEditorDetails::setEnabledAttendeeInput( bool enabled )
{
  mNameEdit->setEnabled( enabled );
  mEmailEdit->setEnabled( enabled );
  mRoleCombo->setEnabled( enabled );
  mStatusCombo->setEnabled( enabled );
  mRsvpButton->setEnabled( enabled );

  mRemoveButton->setEnabled( enabled );
}

void KOEditorDetails::updateAttendeeItem()
{
  if (mDisableItemUpdate) return;

  QListViewItem *item = mGanttView->selectedItem();
  AttendeeListItem *aItem = static_cast<AttendeeListItem *>( item );
  if ( !aItem ) return;

  Attendee *a = aItem->data();

  a->setName( mNameEdit->text() );
  a->setUid( mUidEdit->text() );
  a->setEmail( mEmailEdit->text() );
  a->setRole( Attendee::Role( mRoleCombo->currentItem() ) );
  a->setStatus( Attendee::PartStat( mStatusCombo->currentItem() ) );
  a->setRSVP( mRsvpButton->isChecked() );
  aItem->updateItem();
  updateStatusSummary();
}


/*!
  This slot is called when the user changes the email address of a
  participant. It downloads the free/busy data from the net and enters
  it into the Gantt view by means of the KOGroupware class.
*/
void KOEditorDetails::updateFreeBusyData()
{
  // kdDebug(5850) << "KOEditorDetails::updateFreeBusyData()" << endl;
  if( KOGroupware::instance() ) {
    AttendeeListItem* item = static_cast<AttendeeListItem*>( mGanttView->selectedItem() );
    if( !item ) return; // protect against spurious signals
    QString email = item->listViewText(1);
    if( email == KOPrefs::instance()->email() ) {
      // This could probably be done in a nicer fashion, but we don't want to
      // download our own free-busy list from the net
      QString fbText = KOGroupware::instance()->getFreeBusyString();
      slotInsertFreeBusy( email, KOGroupware::instance()->parseFreeBusy( fbText.utf8() ) );
    } else
      KOGroupware::instance()->downloadFreeBusyData( email,
						     this, SLOT( slotInsertFreeBusy( const QString&, FreeBusy* ) ) );
  }
}

// Set the Free Busy list for everyone having this email address
// If fb == 0, this disabled the free busy list for them
void KOEditorDetails::slotInsertFreeBusy( const QString& email, FreeBusy* fb )
{
  if( fb )
    fb->sortList();
  bool block = mGanttView->getUpdateEnabled();
  mGanttView->setUpdateEnabled(false);
  for( KDGanttViewItem* it = mGanttView->firstChild(); it; it = it->nextSibling() ) {
    AttendeeListItem* item = static_cast<AttendeeListItem*>( it );
    if( item->email() == email )
      item->setFreeBusyPeriods( fb );
  }
  mGanttView->setUpdateEnabled(block);
}


/*!
  Centers the Gantt view to the date/time passed in.
*/

void KOEditorDetails::slotUpdateGanttView( QDateTime dtFrom, QDateTime dtTo )
{
  // kdDebug(5850) << "Center timeline from " << dtFrom.toString() << " to " << dtTo.toString() << endl;
  bool block = mGanttView->getUpdateEnabled( );
  mGanttView->setUpdateEnabled( false );
  QDateTime horizonStart = QDateTime( dtFrom.addDays( -15 ).date() );
  mGanttView->setHorizonStart( horizonStart  );
  mGanttView->setHorizonEnd( dtTo.addDays( 15 ) );
  mGanttView->clearBackgroundColor();
  mGanttView->setIntervalBackgroundColor( dtFrom, dtTo, Qt::magenta ); 
  mGanttView->setUpdateEnabled( block );
  mGanttView->centerTimelineAfterShow( dtFrom );
}


/*!
  This slot is called when the user clicks the "Pick a date" button.
*/
void KOEditorDetails::slotPickDate()
{
  QDateTime start = mEventTimeWidget->startDateTime();
  QDateTime end = mEventTimeWidget->endDateTime();
  bool success = findFreeSlot( start, end );

  if( success ) {
    if ( start == mEventTimeWidget->startDateTime() &&   end  == mEventTimeWidget->endDateTime()  ) {
      KMessageBox::information( this, i18n( "The meeting has already suitable start/end times" ));
    } else {
      mEventTimeWidget->slotDateTimesChanged( start, end, "", true );
      KMessageBox::information( this, i18n( "The meeting has been moved to\nStart: %1\nEnd: %2." ).arg( start.toString() ).arg( end.toString() ) );
    }
  } else
    KMessageBox::sorry( this, i18n( "No suitable date found" ) );
}


/*!
  Finds a free slot in the future which has at least the same size as
  the initial slot.
*/
bool KOEditorDetails::findFreeSlot( QDateTime& dtFrom, QDateTime& dtTo )
{
  // kdDebug(5850) << "+++Find a free slot for " << dtFrom.toString() << " to " << dtTo.toString() << endl;

  QDateTime tryFrom = dtFrom;
  QDateTime tryTo = dtTo;

  // Make sure that we never suggest a date in the past, even if the
  // user originally scheduled the meeting to be in the past.
  if( tryFrom < QDateTime::currentDateTime() ) {
    // The slot to look for is at least partially in the past.
    int secs = tryFrom.secsTo( tryTo );
    tryFrom = QDateTime::currentDateTime();
    tryTo = tryFrom.addSecs( secs );
  }

  bool found = false;
  while( !found ) {
    found = tryDate( tryFrom, tryTo );
    // PENDING(kalle) Make the interval configurable
    if( !found && dtFrom.daysTo( tryFrom ) > 365 )
      break; // don't look more than one year in the future
  }

  dtFrom = tryFrom;
  dtTo = tryTo;

  return found;
}


/*!
  Checks whether the slot specified by (tryFrom, tryTo) is free
  for all participants. If yes, return true. If at least one
  participant is found for which this slot is occupied, this method
  returns false, and (tryFrom, tryTo) contain the next free slot for
  that participant. In other words, the returned slot does not have to
  be free for everybody else.
*/
bool KOEditorDetails::tryDate( QDateTime& tryFrom, QDateTime& tryTo )
{
  // kdDebug(5850) << "++++try " << tryFrom.toString() << " to " << tryTo.toString() << endl;

  AttendeeListItem* currentItem = static_cast<AttendeeListItem*>( mGanttView->firstChild() );
  while( currentItem ) {
    if( !tryDate( currentItem, tryFrom, tryTo ) ) {
      // kdDebug(5850) << "++++date is not OK, new suggestion: " << tryFrom.toString() << " to " << tryTo.toString() << endl;
      return false;
    }

    currentItem = static_cast<AttendeeListItem*>( currentItem->nextSibling() );
  }

  // kdDebug(5850) << "++++date is OK" << endl;
  return true;
}

/*!
  Checks whether the slot specified by (tryFrom, tryTo) is available
  for the participant specified with attendee. If yes, return true. If
  not, return false and change (tryFrom, tryTo) to contain the next
  possible slot for this participant (not necessarily a slot that is
  available for all participants).
*/
bool KOEditorDetails::tryDate( AttendeeListItem* attendee,
                               QDateTime& tryFrom, QDateTime& tryTo )
{
  // kdDebug(5850) << "+++++try " << tryFrom.toString() << " to " << tryTo.toString() << " for " << attendee->listViewText( 0 ) << endl;

  // If we don't have any free/busy information, assume the
  // participant is free. Otherwise a participant without available
  // information would block the whole allocation.
  KCal::FreeBusy* fb = attendee->freeBusy();
  if( !fb )
    return true;

  QValueList<KCal::Period> busyPeriods = fb->busyPeriods();
  for( QValueList<KCal::Period>::Iterator it = busyPeriods.begin();
       it != busyPeriods.end(); ++it ) {
    if( (*it).end() <= tryFrom || // busy period ends before try period
	(*it).start() >= tryTo )  // busy period starts after try period
      continue;
    else {
      // the current busy period blocks the try period, try
      // after the end of the current busy period
      int secsDuration = tryFrom.secsTo( tryTo );
      tryFrom = (*it).end();
      tryTo = tryFrom.addSecs( secsDuration );
      // try again with the new try period (tail-recursive call,
      // a Scheme interpreter would optimize the recursion away
      // :-))
      tryDate( attendee, tryFrom, tryTo );
      // we had to change the date at least once
      // kdDebug(5850) << "+++++slot is not OK, new suggestion: " << tryFrom.toString() << " to " << tryTo.toString() << endl;
      return false;
    }
  }

  // kdDebug(5850) << "+++++slot is OK" << endl;
  return true;
}

void KOEditorDetails::updateStatusSummary()
{
  AttendeeListItem *aItem = static_cast<AttendeeListItem*>(mGanttView->firstChild());
  int total = 0;
  int accepted = 0;
  int tentative = 0;
  int declined = 0;
  while( aItem ) {
    ++total;
    switch( aItem->data()->status() ) {
    case Attendee::Accepted:
      ++accepted;
      break;
    case Attendee::Tentative:
      ++tentative;
      break;
    case Attendee::Declined:
      ++declined;
      break;
    case Attendee::NeedsAction:
    case Attendee::Delegated:
    case Attendee::Completed:
    case Attendee::InProcess:
      /* just to shut up the compiler */
      break;
    }
    aItem = static_cast<AttendeeListItem*>(aItem->nextSibling());
  }
  if( total > 1 && mIsOrganizer ) {
    mStatusSummaryLabel->show();
    mStatusSummaryLabel->setText( i18n( "Of the %1 participants, %2 have accepted, %3"
					" have tentatively accepted, and %4 have declined.")
				  .arg(total).arg(accepted).arg(tentative).arg(declined));
  } else {
    mStatusSummaryLabel->hide();
    mStatusSummaryLabel->setText("");
  }
  mStatusSummaryLabel->adjustSize();
}
