/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file is a test script.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgtestmacro.h"

/**
 * The main function of the unit test
 * @param argc the number of arguments
 * @param argv the list of arguments
 */
int main(int argc, char** argv)
{
    Q_UNUSED(argc);
    Q_UNUSED(argv);

    // Init test
    SKGINITTEST(true);

    // Test class SKGError
    {
        SKGTraces::cleanProfilingStatistics();

        SKGError err;
        SKGTEST("ERR:Default RC", err.getReturnCode(), 0);
        SKGTEST("ERR:Default RC", err.getReturnCode(), 0);
        SKGTESTBOOL("ERR:isWarning", err.isWarning(), false);
        SKGTESTBOOL("ERR:isSucceeded", err.isSucceeded(), true);
        SKGTESTBOOL("ERR:isFailed", err.isFailed(), false);
        SKGTESTBOOL("ERR:!", !err, true);
        SKGTESTBOOL("ERR:operator bool", err, false);
        SKGTESTBOOL("ERR:getPreviousError", (err.getPreviousError() == NULL), true);

        SKGTEST("ERR:Default message", err.getMessage(), "");
        err.setReturnCode(10);
        err.setMessage("Invalid parameter");
        SKGTEST("ERR:getHistoricalSize", err.getHistoricalSize(), 0);
        SKGTEST("ERR:RC 10", err.getReturnCode(), 10);
        SKGTEST("ERR:MSG Invalid parameter", err.getMessage(), "Invalid parameter");
        err.addError(11, "Message 11");
        SKGTESTBOOL("ERR:isWarning", err.isWarning(), false);
        SKGTESTBOOL("ERR:isSucceeded", err.isSucceeded(), false);
        SKGTESTBOOL("ERR:isFailed", err.isFailed(), true);
        SKGTESTBOOL("ERR:!", !err, false);
        SKGTESTBOOL("ERR:operator bool", err, true);

        SKGTEST("ERR:getHistoricalSize", err.getHistoricalSize(), 1);
        err.addError(-12, "Message 12");
        SKGTESTBOOL("ERR:isWarning", err.isWarning(), true);
        SKGTESTBOOL("ERR:isSucceeded", err.isSucceeded(), true);
        SKGTESTBOOL("ERR:isFailed", err.isFailed(), false);
        SKGTESTBOOL("ERR:!", !err, true);
        SKGTESTBOOL("ERR:operator bool", err, false);
        SKGTEST("ERR:getHistoricalSize", err.getHistoricalSize(), 2);
        SKGTEST("ERR:getFullMessageWithHistorical", err.getFullMessageWithHistorical(), "[WAR--12]: Message 12\n[ERR-11]: Message 11\n[ERR-10]: Invalid parameter");
    }

    {
        SKGTraces::cleanProfilingStatistics();

        SKGError err;
        err.setReturnCode(10).setMessage("Invalid parameter").addError(11, "Message 11");
        SKGTEST("ERR:getHistoricalSize", err.getHistoricalSize(), 1);
        SKGTEST("ERR:RC 10", err.getReturnCode(), 11);
        SKGTEST("ERR:MSG Message 11", err.getMessage(), "Message 11");
    }

    // SKGServices
    {
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("2014"), "STRFTIME('%Y',d_date)='2014'");
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("2014-02"), "STRFTIME('%Y-%m',d_date)='2014-02'");
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("2014-10"), "STRFTIME('%Y-%m',d_date)='2014-10'");
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("2014-Q2"), "STRFTIME('%Y',d_date)||'-Q'||(CASE WHEN STRFTIME('%m',d_date)<='03' THEN '1' WHEN STRFTIME('%m',d_date)<='06' THEN '2' WHEN STRFTIME('%m',d_date)<='09' THEN '3' ELSE '4' END)='2014-Q2'");
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("2014-S2"), "STRFTIME('%Y',d_date)||'-S'||(CASE WHEN STRFTIME('%m',d_date)<='06' THEN '1' ELSE '2' END)='2014-S2'");
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("INVALID STRING"), "1=0");
        SKGTEST("CONV:getPeriodWhereClause", SKGServices::getPeriodWhereClause("ALL"), "1=1");

        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014"), "2013");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014-10"), "2014-09");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014-01"), "2013-12");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014-Q2"), "2014-Q1");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014-Q1"), "2013-Q4");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014-S2"), "2014-S1");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("2014-S1"), "2013-S2");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("INVALID STRING"), "1=0");
        SKGTEST("CONV:getPerviousPeriod", SKGServices::getPerviousPeriod("ALL"), "1=0");

        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014").toString(), QDate(2014, 12, 31).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-S1").toString(), QDate(2014, 6, 30).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-S2").toString(), QDate(2014, 12, 31).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-Q1").toString(), QDate(2014, 3, 31).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-Q2").toString(), QDate(2014, 6, 30).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-Q3").toString(), QDate(2014, 9, 30).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-Q4").toString(), QDate(2014, 12, 31).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-01").toString(), QDate(2014, 1, 31).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("2014-07").toString(), QDate(2014, 7, 31).toString());
        SKGTEST("CONV:periodToDate", SKGServices::periodToDate("ALL").toString(), QDate(0, 1, 1).toString());

        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("INVALID", true).toString(), "");

        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2", true).toString(), QDate(QDate::currentDate().year(), 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/14", true).toString(), QDate(2014, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/99", true).toString(), QDate(1999, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/014", true).toString(), QDate(2014, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/199", true).toString(), QDate(1199, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/1014", true).toString(), QDate(1014, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/2222", true).toString(), QDate(2222, 2, 1).toString());

        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2", false).toString(), QDate(QDate::currentDate().year() + 1, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/14", false).toString(), QDate(2114, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/99", false).toString(), QDate(2099, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/014", false).toString(), QDate(3014, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/199", false).toString(), QDate(2199, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/1014", false).toString(), QDate(1014, 2, 1).toString());
        SKGTEST("CONV:partialStringToDate", SKGServices::partialStringToDate("1/2/2222", false).toString(), QDate(2222, 2, 1).toString());

        SKGTEST("CONV:dateToPeriod", SKGServices::dateToPeriod(QDate(2013, 03, 05), "D"), "2013-03-05");
        SKGTEST("CONV:dateToPeriod", SKGServices::dateToPeriod(QDate(2013, 03, 05), "W"), "2013-W10");
        SKGTEST("CONV:dateToPeriod", SKGServices::dateToPeriod(QDate(2013, 03, 05), "M"), "2013-03");
        SKGTEST("CONV:dateToPeriod", SKGServices::dateToPeriod(QDate(2013, 03, 05), "Q"), "2013-Q1");
        SKGTEST("CONV:dateToPeriod", SKGServices::dateToPeriod(QDate(2013, 03, 05), "S"), "2013-S1");
        SKGTEST("CONV:dateToPeriod", SKGServices::dateToPeriod(QDate(2013, 03, 05), "Y"), "2013");

        SKGTEST("CONV:intToString", SKGServices::intToString(10), "10");
        SKGTEST("CONV:intToString", SKGServices::intToString(3.5), "3");
        SKGTEST("CONV:intToString", SKGServices::intToString(5490990004), "5490990004");
        SKGTEST("CONV:stringToInt", SKGServices::stringToInt("56"), 56);
        SKGTEST("CONV:stringToInt", SKGServices::stringToInt("HELLO"), 0);
        SKGTEST("CONV:stringToInt", SKGServices::stringToInt("5HELLO"), 0);
        SKGTEST("CONV:stringToInt", SKGServices::stringToInt("5490990004"), 5490990004);
        SKGTEST("CONV:doubleToString", SKGServices::doubleToString(10), "10");
        SKGTEST("CONV:doubleToString", SKGServices::doubleToString(5.3), "5.3");
        SKGTEST("CONV:doubleToString", SKGServices::doubleToString(11111115.33), "11111115.33");
        SKGTEST("CONV:doubleToString", SKGServices::doubleToString(119999.11), "119999.11");
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("10") - 10, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("5.3") - 5.3, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("11111115.33") - 11111115.33, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("25,000.00") - 25000.00, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("1.307,40") - 1307.40, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("2 150,10") - 2150.10, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("-$8.35") + 8.35, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("8.35€") - 8.35, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("1234.56e-02") - 12.3456, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("31238/100") - 312.38, 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("nan"), 0);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("inf"), 1e300);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("-inf"), -1e300);
        SKGTEST("CONV:stringToDouble", SKGServices::stringToDouble("00000000194065"), 194065);
        SKGServices::timeToString(QDateTime());
        SKGTEST("CONV:stringToTime", SKGServices::timeToString(SKGServices::stringToTime("1970-07-16")), "1970-07-16 00:00:00");
        SKGTEST("CONV:stringToTime", SKGServices::timeToString(SKGServices::stringToTime("2008-04-20")), "2008-04-20 00:00:00");
        SKGTEST("CONV:stringToTime", SKGServices::dateToSqlString(SKGServices::stringToTime("1970-07-16")), "1970-07-16");
        SKGTEST("CONV:stringToTime", SKGServices::dateToSqlString(SKGServices::stringToTime("2008-04-20")), "2008-04-20");

        SKGTESTBOOL("CONV:SKGServices::getMicroTime", (SKGServices::getMicroTime() > 0), true);

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("20080525", "YYYYMMDD"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("2008-05-25", "YYYY-MM-DD"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("2008-05-25 00:00:00", "YYYY-MM-DD"), "2008-05-25");

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("05/25/08", "MM/DD/YY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("05/25/78", "MM/DD/YY"), "1978-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("05-25-08", "MM-DD-YY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("05-25-78", "MM-DD-YY"), "1978-05-25");

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5/25/08", "MM/DD/YY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5/25/78", "MM/DD/YY"), "1978-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5-25-08", "MM-DD-YY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5-6-08", "MM-DD-YY"), "2008-05-06");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5-25-78", "MM-DD-YY"), "1978-05-25");

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("25/05/08", "DD/MM/YY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("25/05/78", "DD/MM/YY"), "1978-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("25-05-08", "DD-MM-YY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("25-05-78", "DD-MM-YY"), "1978-05-25");

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("25/05/2008", "DD/MM/YYYY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("25-05-2008", "DD-MM-YYYY"), "2008-05-25");

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5/25/2008", "MM/DD/YYYY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5-25-2008", "MM-DD-YYYY"), "2008-05-25");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5-6-2008", "MM-DD-YYYY"), "2008-05-06");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("5-6-8", "MM-DD-YYYY"), "2008-05-06");

        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("3.21.01", "MM-DD-YY"), "2001-03-21");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("1/ 1' 6", "DD-MM-YY"), "2006-01-01");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("6/ 1/94", "DD-MM-YY"), "1994-01-06");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("21/12'2001", "DD-MM-YYYY"), "2001-12-21");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("21122001", "DDMMYYYY"), "2001-12-21");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("12212001", "MMDDYYYY"), "2001-12-21");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("010203", "MMDDYY"), "2003-01-02");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("010203", "DDMMYY"), "2003-02-01");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("3/9/04", "DD-MM-YY"), "2004-09-03");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31Dec2005", "DDMMMYYYY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31-Dec-2005", "DD-MMM-YYYY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31/Dec/2005", "DD/MMM/YYYY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31Dec05", "DDMMMYY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31-Dec-05", "DD-MMM-YY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31/Dec/05", "DD/MMM/YY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("31DEC2005", "DDMMMYYYY"), "2005-12-31");
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("INVALIDDATE", "DD-MM-YY"), SKGServices::dateToSqlString(QDateTime::currentDateTime()));
        SKGTEST("STR:dateToSqlString", SKGServices::dateToSqlString("02.01.2015", "DD-MM-YYYY"), "2015-01-02");


        SKGTEST("DATE:nbWorkingDays", SKGServices::nbWorkingDays(QDate(2010, 9, 3), QDate(2010, 9, 6)), 1);
        SKGTEST("DATE:nbWorkingDays", SKGServices::nbWorkingDays(QDate(2010, 9, 6), QDate(2010, 9, 3)), 1);
        SKGTEST("DATE:nbWorkingDays", SKGServices::nbWorkingDays(QDate(2010, 9, 3), QDate(2010, 9, 3)), 1);
        SKGTEST("DATE:nbWorkingDays", SKGServices::nbWorkingDays(QDate(2010, 9, 1), QDate(2010, 9, 3)), 2);
        SKGTEST("DATE:nbWorkingDays", SKGServices::nbWorkingDays(QDate(2010, 9, 1), QDate(2010, 9, 8)), 5);

        {
            QStringList dates;
            dates << "19/08/2008";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YYYY");
        }
        {
            QStringList dates;
            dates << "20080819";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "YYYYMMDD");
        }
        {
            QStringList dates;
            dates << "10141989";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "MMDDYYYY");
        }
        {
            QStringList dates;
            dates << "14101989";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DDMMYYYY");
        }
        {
            QStringList dates;
            dates << "05/08/2008" << "19/08/2008";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YYYY");
        }
        {
            QStringList dates;
            dates << "19/ 1' 6";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YYYY");
        }
        {
            QStringList dates;
            dates << "21/12'2001";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YYYY");
        }
        {
            QStringList dates;
            dates << "6/ 1/94";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YYYY");
        }
        {
            QStringList dates;
            dates << "10.29.07";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "MM-DD-YY");
        }
        {
            QStringList dates;
            dates << "10.05.07";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YY");
        }
        {
            QStringList dates;
            dates << "2001.10.25";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "YYYY-MM-DD");
        }
        {
            QStringList dates;
            dates << "45450116";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "YYYYMMDD");
        }
        {
            QStringList dates;
            dates << "7/14' 0" << "11/30/99";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "MM-DD-YYYY");
        }
        {
            QStringList dates;
            dates << "10/ 8'10";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YYYY");
        }
        {
            QStringList dates;
            dates << "7/8/06";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YY");
        }
        {
            QStringList dates;
            dates << "7/8/06" << "11/30/99" << "7/14' 0";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "MM-DD-YYYY");
        }
        {
            QStringList dates;
            dates << "3/9/04";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YY");
        }
        {
            QStringList dates;
            dates << "31Dec2005";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DDMMMYYYY");
        }
        {
            QStringList dates;
            dates << "31-Dec-05";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MMM-YY");
        }
        {
            QStringList dates;
            dates << "2008-05-25 00:00:00";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "YYYY-MM-DD");
        }
        {
            QStringList dates;
            QFile inputFile(SKGTest::getTestPath("IN") % "/dates.txt");
            if (inputFile.open(QIODevice::ReadOnly)) {
                QTextStream in(&inputFile);
                while (!in.atEnd()) {
                    QString l = in.readLine().trimmed();
                    if (!l.isEmpty()) {
                        dates << l;
                    }
                }

                inputFile.close();
            }
            SKGTEST("STR:dats.count", dates.count(), 2364);
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "MM-DD-YY");
        }
        {
            QStringList dates;
            dates << "  7/8/06    " << "";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "DD-MM-YY");
        }
        {
            QStringList dates;
            dates << "99999999" << "9999999999";
            SKGTEST("STR:getDateFormat", SKGServices::getDateFormat(dates), "");
        }
        SKGTEST("STR:stringToCsv", SKGServices::stringToCsv("ABC"), "\"ABC\"");
        SKGTEST("STR:stringToCsv", SKGServices::stringToCsv("ABC;CDE"), "\"ABC;CDE\"");
        SKGTEST("STR:stringToCsv", SKGServices::stringToCsv("AB \"C\" DE"), "\"AB \"\"C\"\" DE\"");
        SKGTEST("STR:stringToCsv", SKGServices::stringToCsv("AB \"C\";DE"), "\"AB \"\"C\"\";DE\"");

        QStringList parameters = SKGServices::splitCSVLine("52.33,\"9/28/2010\",52.36,231803,52.33,0.00,+0.15,-,0.00,-,0.00,0.00,0.00,0.00,\"- - -\",\"-\",-,\"n\",N/A,0,+15.82,+43.33%,9,672,-1.08,-2.02%,-,-,\"51.76 - 52.57\",0.00,0.00,N/A,N/A,N/A,N/A,\"-\",51.91,52.18,-,\"+0.29%\",N/A,N/A,\"N/A\",N/A,\"N/A\",N/A,N/A,N/A,-,N/A,\"11:35am\",N/A,211524,-,\"36.51 - 53.41\",\"- - +0.29%\",\"Paris\",N/A,\"DASSAULT SYST.\"", ',');
        SKGTEST("STR:splitCSVLine count", parameters.count(), 59);

        parameters = SKGServices::splitCSVLine("2013-04-02;transfer;\"a2\";'some text 2';-20,13", ';', true);
        SKGTEST("STR:splitCSVLine count", parameters.count(), 5);
        SKGTEST("STR:splitCSVLine", parameters[0], "2013-04-02");
        SKGTEST("STR:splitCSVLine", parameters[1], "transfer");
        SKGTEST("STR:splitCSVLine", parameters[2], "a2");
        SKGTEST("STR:splitCSVLine", parameters[3], "some text 2");
        SKGTEST("STR:splitCSVLine", parameters[4], "-20,13");

        parameters = SKGServices::splitCSVLine("\"A;'B\";'A;\"B'", ';', true);
        SKGTEST("STR:splitCSVLine count", parameters.count(), 2);
        SKGTEST("STR:splitCSVLine", parameters[0], "A;'B");
        SKGTEST("STR:splitCSVLine", parameters[1], "A;\"B");

        SKGTEST("STR:stringToHtml", SKGServices::stringToHtml("<hello>&<world>"), "&lt;hello&gt;&amp;&lt;world&gt;");
        SKGTEST("STR:htmlToString", SKGServices::htmlToString("&lt;hello&gt;&amp;&lt;world&gt;"), "<hello>&<world>");

        parameters = SKGServices::splitCSVLine(SKGServices::stringToCsv("Y") % ';' % SKGServices::stringToCsv("A;B"));
        SKGTEST("STR:splitCSVLine count", parameters.count(), 2);
        SKGTEST("STR:splitCSVLine", parameters[0], "Y");
        SKGTEST("STR:splitCSVLine", parameters[1], "A;B");

        parameters = SKGServices::splitCSVLine("A;\"B;C\";D");
        SKGTEST("STR:splitCSVLine count", parameters.count(), 3);
        SKGTEST("STR:splitCSVLine", parameters[0], "A");
        SKGTEST("STR:splitCSVLine", parameters[1], "B;C");
        SKGTEST("STR:splitCSVLine", parameters[2], "D");

        parameters = SKGServices::splitCSVLine("'A';'B'");
        SKGTEST("STR:splitCSVLine count", parameters.count(), 2);
        SKGTEST("STR:splitCSVLine", parameters[0], "A");
        SKGTEST("STR:splitCSVLine", parameters[1], "B");

        parameters = SKGServices::splitCSVLine("\"A ' B\",\"C\"");
        SKGTEST("STR:splitCSVLine count", parameters.count(), 2);
        SKGTEST("STR:splitCSVLine", parameters[0], "A ' B");
        SKGTEST("STR:splitCSVLine", parameters[1], "C");

        parameters = SKGServices::splitCSVLine("'A \" B','C'");
        SKGTEST("STR:splitCSVLine count", parameters.count(), 2);
        SKGTEST("STR:splitCSVLine", parameters[0], "A \" B");
        SKGTEST("STR:splitCSVLine", parameters[1], "C");

        QChar realsep;
        parameters = SKGServices::splitCSVLine("\"A\",18,\"B\"", ';', true, &realsep);
        SKGTEST("STR:splitCSVLine count", parameters.count(), 3);
        SKGTEST("STR:splitCSVLine", parameters[0], "A");
        SKGTEST("STR:splitCSVLine", parameters[1], "18");
        SKGTEST("STR:splitCSVLine", parameters[2], "B");
        SKGTEST("STR:sep", realsep, ',');

        parameters = SKGServices::splitCSVLine("30/05/2008;RETRAIT ESPECES AGENCE;                                ;      100,00;DEBIT;");
        SKGTEST("STR:splitCSVLine count", parameters.count(), 6);

        parameters = SKGServices::splitCSVLine("A|\"B\";\"C\"", '|', false);
        SKGTEST("STR:splitCSVLine count", parameters.count(), 2);
        SKGTEST("STR:splitCSVLine", parameters[0], "A");
        SKGTEST("STR:splitCSVLine", parameters[1], "\"B\";\"C\"");

        parameters = SKGServices::splitCSVLine("+123 \"-abc def\" \"e:f\" e:f", ' ', true);
        SKGTEST("STR:splitCSVLine count", parameters.count(), 4);
        SKGTEST("STR:splitCSVLine", parameters[0], "+123");
        SKGTEST("STR:splitCSVLine", parameters[1], "-abc def");
        SKGTEST("STR:splitCSVLine", parameters[2], "e:f");
        SKGTEST("STR:splitCSVLine", parameters[3], "e:f");

        SKGTEST("STR:splitCSVLine words", SKGServices::splitCSVLine("w1 w2", ' ', true).count(), 2);
        SKGTEST("STR:splitCSVLine words", SKGServices::splitCSVLine("\"w1 w1\"", ' ', true).count(), 1);
        SKGTEST("STR:splitCSVLine words", SKGServices::splitCSVLine("\"w1 w1\" w2 \"w3 w3 w3\" w4", ' ', true).count(), 4);

        QList< SKGServices::SKGSearchCriteria > criterias = SKGServices::stringToSearchCriterias("abc +def +ghi");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 3);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "ghi");
        SKGTEST("STR:stringToSearchCriterias mode", criterias[1].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[1].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[1].words[0], "def");
        SKGTEST("STR:stringToSearchCriterias mode", criterias[2].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[2].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[2].words[0], "abc");

        criterias = SKGServices::stringToSearchCriterias("-abc +def ghi");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 2);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 2);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "def");
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[1], "ghi");
        SKGTEST("STR:stringToSearchCriterias mode", criterias[1].mode, '-');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[1].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[1].words[0], "abc");

        criterias = SKGServices::stringToSearchCriterias("-abc +def ghi -10");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 2);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 3);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "def");
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[1], "ghi");
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[2], "-10");
        SKGTEST("STR:stringToSearchCriterias mode", criterias[1].mode, '-');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[1].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[1].words[0], "abc");

        criterias = SKGServices::stringToSearchCriterias("-10");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 1);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "-10");

        criterias = SKGServices::stringToSearchCriterias("-abc");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 2);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "");
        SKGTEST("STR:stringToSearchCriterias mode", criterias[1].mode, '-');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[1].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[1].words[0], "abc");

        criterias = SKGServices::stringToSearchCriterias("\"abc def ghi\" \"123 456\"");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 1);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 2);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "abc def ghi");
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[1], "123 456");

        criterias = SKGServices::stringToSearchCriterias("\"-payee:abc def : ghi\" +amount:25");
        SKGTEST("STR:stringToSearchCriterias count", criterias.count(), 2);
        SKGTEST("STR:stringToSearchCriterias mode", criterias[0].mode, '+');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[0].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[0].words[0], "amount:25");
        SKGTEST("STR:stringToSearchCriterias mode", criterias[1].mode, '-');
        SKGTEST("STR:stringToSearchCriterias words count", criterias[1].words.count(), 1);
        SKGTEST("STR:stringToSearchCriterias word", criterias[1].words[0], "payee:abc def : ghi");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(d_DATEOP) LIKE '%toto%' OR lower(i_number) LIKE '%toto%' OR lower(t_PAYEE) LIKE '%toto%' OR lower(t_bookmarked) LIKE '%toto%'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("i_num:ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(i_number) LIKE '%toto%'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("i_num=1234"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((i_number=1234))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("t_PAYEE>ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(t_PAYEE)>'toto'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("t_PAYEE<ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(t_PAYEE)<'toto'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("t_PAYEE>=ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(t_PAYEE)>='toto'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("t_PAYEE<=ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(t_PAYEE)<='toto'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("t_PAYEE=ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(t_PAYEE)='toto'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("t_PAYEE#^t[o|a]to$"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((REGEXP('^t[o|a]to$',t_PAYEE)))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("+i_number<20 +i_number>30"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((i_number>30)) OR ((i_number<20))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("d_DATEOP>2015-05-04"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "((lower(d_DATEOP)>'2015-05-04'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("notfound:ToTo"),
                        QStringList() << "d_DATEOP" << "i_number" << "t_PAYEE" << "t_bookmarked"),
                "(1=0)");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias("v1 v2 +v3 -v4 -v5"),
                        QStringList() << "t_comment"),
                "((lower(t_comment) LIKE '%v3%')) OR ((lower(t_comment) LIKE '%v1%') AND (lower(t_comment) LIKE '%v2%')) AND NOT((lower(t_comment) LIKE '%v4%')) AND NOT((lower(t_comment) LIKE '%v5%'))");

        SKGTEST("STR:searchCriteriasToWhereClause",
                SKGServices::searchCriteriasToWhereClause(SKGServices::stringToSearchCriterias(":ToTo"),
                        QStringList() << "p_prop1" << "p_prop2"),
                "((i_PROPPNAME='prop1' AND (lower(i_PROPVALUE) LIKE '%toto%') OR i_PROPPNAME='prop2' AND (lower(i_PROPVALUE) LIKE '%toto%')))");

        SKGStringListList table;
        table.push_back(QStringList() << "Person" << "Salary" << "Age");
        table.push_back(QStringList() << "John" << "58000" << "33");
        table.push_back(QStringList() << "Paul" << "42000" << "25");
        table.push_back(QStringList() << "Alan" << "65000" << "41");
        SKGStringListList table100 = SKGServices::getBase100Table(table);

        SKGStringListList tablePourcent = SKGServices::getPercentTable(table, true, false);
        SKGStringListList tablePourcent2 = SKGServices::getPercentTable(table, false, true);

        SKGStringListList tableHistory = SKGServices::getHistorizedTable(table);

        SKGTEST("STR:encodeForUrl", SKGServices::encodeForUrl("abc"), "abc");

        SKGTEST("STR:getMajorVersion", SKGServices::getMajorVersion("4.3.12.3"), "4.3");
        SKGTEST("STR:getMajorVersion", SKGServices::getMajorVersion("4.3.12"), "4.3");
        SKGTEST("STR:getMajorVersion", SKGServices::getMajorVersion("4.3"), "4.3");
        SKGTEST("STR:getMajorVersion", SKGServices::getMajorVersion("4"), "4");

        SKGTEST("STR:toCurrencyString", SKGServices::toCurrencyString(5.12341234, "F", 2), "F 5.12");
        SKGTEST("STR:toCurrencyString", SKGServices::toCurrencyString(5.12341234, "F", 4), "F 5.1234");

        SKGTEST("STR:toPercentageString", SKGServices::toPercentageString(5.12341234, 2), "5.12 %");
        SKGTEST("STR:toPercentageString", SKGServices::toPercentageString(5.12341234, 4), "5.1234 %");
    }

    // End test
    SKGENDTEST();
}
