/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGOPERATIONOBJECT_H
#define SKGOPERATIONOBJECT_H
/** @file
 * This file defines classes SKGOperationObject.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */

#include "skgobjectbase.h"
#include "skgbankmodeler_export.h"
class SKGAccountObject;
class SKGUnitObject;
class SKGSubOperationObject;
class SKGRecurrentOperationObject;
class SKGPayeeObject;

/**
 * This class manages operation object
 */
class SKGBANKMODELER_EXPORT SKGOperationObject : public SKGObjectBase
{
    Q_OBJECT
    /**
     * This enumerate defines status for operations
     */
    Q_ENUMS(OperationStatus)
public:
    /**
     * This enumerate defines status for operations
     */
    enum OperationStatus {NONE, /**< no status */
                          POINTED, /**< pointed */
                          CHECKED /**< checked */
                         };

    /**
    * This enumerate defines the alignment amount mode
    */
    enum AmountAlignmentMode {
        DEFAULT,         /**< Default */
        PROPORTIONAL,    /**< Proportional */
        ADDSUBOPERATION  /**< Add sub operation */
    };

    /**
     * Constructor
     * @param iDocument the document containing the object
     * @param iID the identifier in @p iTable of the object
     */
    explicit SKGOperationObject(SKGDocument* iDocument = NULL, int iID = 0);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */
    explicit SKGOperationObject(const SKGObjectBase& iObject);

    /**
     * Copy constructor
     * @param iObject the object to copy
     */

    SKGOperationObject(const SKGOperationObject& iObject);

    /**
     * Operator affectation
     * @param iObject the object to copy
     */
    virtual const SKGOperationObject& operator= (const SKGObjectBase& iObject) Q_DECL_OVERRIDE;

    /**
     * Destructor
     */
    virtual ~SKGOperationObject();

    /**
     * Duplicate current operation including suboperations and grouped operations
     * @param oOperation the created operation
     * @param iDate date(s) for new operation(s)
     * @param iTemplateMode the template mode for new operation(s)
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError duplicate(SKGOperationObject& oOperation, const QDate& iDate = QDate::currentDate(), bool iTemplateMode = false) const;

    /**
     * Get the parent account
     * @param oAccount the parent account
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError getParentAccount(SKGAccountObject& oAccount) const;

    /**
     * Set the parent account
     * @param iAccount the parent account
     * @param iForce force the creation even if the account is closed
     * @return an object managing the error.
     *   @see SKGError
     */
    virtual SKGError setParentAccount(const SKGAccountObject& iAccount, bool iForce = false);

    /**
     * Set the mode of operation
     * @param iNumber the number
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setNumber(qlonglong iNumber);

    /**
     * Get the number of this operation
     * @return the number
     */
    virtual qlonglong getNumber() const;

    /**
     * Set the number of operation
     * @param iMode the mode
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setMode(const QString& iMode);

    /**
     * Get the mode of this operation
     * @return the mode
     */
    virtual QString getMode() const;

    /**
     * Set the payee of operation
     * @param iPayee the payee
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setPayee(const SKGPayeeObject& iPayee);

    /**
     * Get the payee of this operation
     * @param oPayee the payee
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getPayee(SKGPayeeObject& oPayee) const;

    /**
     * Set the comment of operation
     * @param iComment the comment
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setComment(const QString& iComment);

    /**
     * Get the comment of this operation
     * @return the comment
     */
    virtual QString getComment() const;

    /**
     * Get the status of this operation
     * @return the status
     */
    virtual SKGOperationObject::OperationStatus getStatus() const;

    /**
     * Set the status of operation
     * @param iStatus the status
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setStatus(SKGOperationObject::OperationStatus iStatus);

    /**
     * Set date of this operation
     * @param iDate the date
     * @param iRefreshSubOperations to refresh the sub operations
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setDate(const QDate& iDate, bool iRefreshSubOperations = true);

    /**
     * Get date of this operation
     * @return the date
     */
    virtual QDate getDate() const;

    /**
     * Set the unit
     * @param iUnit the unit
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setUnit(const SKGUnitObject& iUnit);

    /**
     * Get the unit
     * @param oUnit the unit
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getUnit(SKGUnitObject& oUnit) const;

    /**
     * To know if an operation is grouped
     * @return true or false
     */
    virtual bool isInGroup() const;

    /**
     * To know if the current operation is a transfer of the other one
     * @param oOperation the other operation
     * @return true or false
     */
    virtual bool isTransfer(SKGOperationObject& oOperation) const;

    /**
     * Set the group operation
     * @param iOperation the operation (itself to remove from group)
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setGroupOperation(const SKGOperationObject& iOperation);

    /**
     * Get the group operation
     * @param oOperation the operation
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getGroupOperation(SKGOperationObject& oOperation) const;

    /**
     * Get all operations in the same group
     * @param oGroupedOperations all operation in the same group
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getGroupedOperations(SKGListSKGObjectBase& oGroupedOperations) const;

    /**
     * To bookmark or not an operation
     * @param iBookmark the bookmark: true or false
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError bookmark(bool iBookmark);

    /**
     * To know if the operation is bookmarked
     * @return an object managing the error
     *   @see SKGError
     */
    virtual bool isBookmarked() const;

    /**
     * To set the imported attribute of an operation
     * @param iImported the imported status: true or false
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setImported(bool iImported);

    /**
     * To know if the operation has been imported or not
     * @return an object managing the error
     *   @see SKGError
     */
    virtual bool isImported() const;

    /**
     * Set the import identifier of operation, t_imported is set to 'T'
     * @param iImportID the import identifier (it's used to check if the operation is already imported)
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setImportID(const QString& iImportID);

    /**
     * Get the import identifier of operation
     * @return the comment
     */
    virtual QString getImportID() const;

    /**
     * To set the template attribute of an operation
     * @param iTemplate the template status: true or false
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError setTemplate(bool iTemplate);

    /**
     * To know if the operation is a template or not
     * @return an object managing the error
     *   @see SKGError
     */
    virtual bool isTemplate() const;

    /**
     * Add a new suboperation to this operation
     * @param oSubOperation the created suboperation
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError addSubOperation(SKGSubOperationObject& oSubOperation);

    /**
     * Get the number of sub operations
     * @return number of sub operations
     */
    virtual int getNbSubOperations() const;

    /**
     * Get all suboperations of this operation
     * @param oSubOperations all suboperations of this operation
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getSubOperations(SKGListSKGObjectBase& oSubOperations) const;

    /**
     * Get the current amount
     * @return the current amount
     */
    virtual double getCurrentAmount() const;

    /**
     * Get the account balance for this operation
     * @return the account balance
     */
    virtual double getBalance() const;

    /**
     * Get amount of the operation at a date
     * @param iDate date
     * @return amount of the operation
     */
    virtual double getAmount(const QDate& iDate) const;

    /**
     * Create a recurrent operation based on this one
     * @param oRecurrentOperation the created recurrent operation
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError addRecurrentOperation(SKGRecurrentOperationObject& oRecurrentOperation) const;

    /**
     * Get the recurrent operations based on this one
     * @param oRecurrentOperation the recurrent operations
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError getRecurrentOperations(SKGListSKGObjectBase& oRecurrentOperation) const;

    /**
     * Merge current operation with another one
     * @param iDeletedOne after merge this operation will be deleted
     * @param iMode the alignment mode
     * @param iSendMessage send warning message
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError mergeAttribute(const SKGOperationObject& iDeletedOne, SKGOperationObject::AmountAlignmentMode iMode = SKGOperationObject::DEFAULT, bool iSendMessage = true);

    /**
     * Merge current operation with another one
     * @param iDeletedOne after merge this operation will be deleted
     * @return an object managing the error
     *   @see SKGError
     */
    virtual SKGError mergeSuboperations(const SKGOperationObject& iDeletedOne);
};
/**
 * Declare the class
 */
Q_DECLARE_TYPEINFO(SKGOperationObject, Q_MOVABLE_TYPE);
#endif
