/********************************************************
*  callbacks.c
*
*  Source file for Matrox video parameters tuning utility.
*  Callback functions for GTK+. First generated by Glade.
*
*  Copyright (C) 2001, Matrox Graphics Inc.
*
*  Author : Stephane Duguay
*  Last modified : April 2001.
*********************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/* GTK lib headers */
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>

/* X11 headers */
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

/* X11 extension lib headers  */
#include <X11/extensions/xf86vmode.h>
#include <X11/extensions/Xinerama.h>
#include <xf86Parser.h>

/* Standard C lib headers */
#include <stdlib.h>
#include <stdio.h>

/* mgaXconf project headers */
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "mgaXaction.h"
#include "mgaXconfig.h"
#include "io.h"
#include "global.h"

/* VERSION */
#define MAJOR 1
#define MINOR 0
#define BUILD 5

/* Max nb. of screens supported by mgaXconf */
#define MAXSCREENS 2

/* Time lenght of identification */
#define IDENTIFY_TIME 3000

/* external global variable */
extern Display* pDisp;
extern XF86VidModeModeLine mode_line;
extern struct DeviceInfo device_info;

/* Screen we are working on */
/* To be improved : not implement with global */
int CurrentScreen = 0;

/* Adjecency where (screens position) */
int where = 0;

/* Global-needed variables */
/* Filenames and struct pointer */
char sConfigFile[255];
char sConfigBackup[255];
XF86ConfigPtr pXF86Config;

/* GLOBAL Window pointer */
GtkWidget* wdgWinAdjust = NULL;
GtkWidget* wdgWinSave = NULL;
GtkWidget* wdgWinCentral = NULL;
GtkWidget* wdgWinIdentify[MAXSCREENS];
GtkWidget* wdgWinOpen = NULL;
GtkWidget* wdgWinResolutions = NULL;
GtkWidget* wdgWinMsgbox = NULL;
GtkWidget* wdgWinPreview = NULL;
GtkWidget* wdgWinInfo = NULL;

/* dirty bits */
BOOL bDidChange = FALSE;
BOOL wasSingleHeadAtStartup = TRUE;
BOOL bHalLoaded = FALSE;
BOOL bDriverModed = FALSE;

/* Saving window - to improve */
#define modeChangeNEXT 1
#define modeChangePREVIOUS -1
#define modeChangeEXIT 0
int modeChange = modeChangeEXIT;

/* helpful local procedures */
static void WriteVideoInfoOnTxtBox(XF86VidModeModeLine* pMode);
static void ValuesCorrection(void);
static void set_window_icon(GtkWidget* this_window);
static void set_window_back_identify(GtkWidget* this_window, GtkWidget* fixed);
static void set_window_on_top(GtkWidget* the_window);
static int QSortCompareBigToSmall(const void* p1, const void* p2);
static int QSortCompareSmallToBig(const void* p1, const void* p2);
static void drawLayout(int scrn1, int confWhere, int scrn2);

/* useful more general procedures */
void MsgBox(char* title, char* msg, int exit);
void set_window_back(GtkWidget* this_window, GtkWidget* fixed);
gboolean timeout_callback_destroy(gpointer pbShown);
int itoa(char *string, int d);
int ftoa(char *string, float f);

/****************************************
* drawLayout places widgets on screen
* according to their logical position.
* 
* IN=>	scrn1 : screen index of first screen.
* 		scrn2 : screen index of second screen.
* 		confWhere : adjacency where position.
* OUT=>	void
*****************************************/
static void drawLayout(int scrn1, int confWhere, int scrn2)
{
	/* hard-coded position of widgets */
	/* Top-left coordonates of resolutions buttons. */
	#define LEFT_X 24
	#define LEFT_Y 96
	#define RIGHT_X 216
	#define RIGHT_Y 96
	#define ABOVE_X 120
	#define ABOVE_Y 16
	#define BELOW_X 120
	#define BELOW_Y 200

	/* Resolutions and Adjust buttons widgets pointers  */
	GtkWidget* wdgButResolution1 = lookup_widget(wdgWinCentral, "buttonResolution1");
    GtkWidget* wdgButResolution2 = lookup_widget(wdgWinCentral, "buttonResolution2");
    GtkWidget* wdgButAdjust1 = lookup_widget(wdgWinCentral, "buttonAdjust1");
    GtkWidget* wdgButAdjust2 = lookup_widget(wdgWinCentral, "buttonAdjust2");

	/* Place widgets according to logical adjacency */
	if(scrn1 == 0 && scrn2 == 1) {
		switch(confWhere) {
			case CONF_ADJ_LEFTOF:
				gtk_widget_set_uposition(wdgButResolution1, LEFT_X, LEFT_Y);
				gtk_widget_set_uposition(wdgButAdjust1,	LEFT_X,	LEFT_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, RIGHT_X, RIGHT_Y);
				gtk_widget_set_uposition(wdgButAdjust2, RIGHT_X, RIGHT_Y + wdgButResolution2->allocation.height);
			break;
			
			case CONF_ADJ_RIGHTOF:
				gtk_widget_set_uposition(wdgButResolution1, RIGHT_X, RIGHT_Y);
				gtk_widget_set_uposition(wdgButAdjust1,	RIGHT_X, RIGHT_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, LEFT_X, LEFT_Y);
				gtk_widget_set_uposition(wdgButAdjust2, LEFT_X, LEFT_Y + wdgButResolution2->allocation.height);
			break;
			
			case CONF_ADJ_ABOVE:
				gtk_widget_set_uposition(wdgButResolution1, ABOVE_X, ABOVE_Y);
				gtk_widget_set_uposition(wdgButAdjust1,	ABOVE_X, ABOVE_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, BELOW_X, BELOW_Y);
				gtk_widget_set_uposition(wdgButAdjust2, BELOW_X, BELOW_Y + wdgButResolution2->allocation.height);
			break;
			
			case CONF_ADJ_BELOW:
				gtk_widget_set_uposition(wdgButResolution1, BELOW_X, BELOW_Y);
				gtk_widget_set_uposition(wdgButAdjust1,	BELOW_X, BELOW_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, ABOVE_X, ABOVE_Y);
				gtk_widget_set_uposition(wdgButAdjust2, ABOVE_X, ABOVE_Y + wdgButResolution2->allocation.height);
			break;
			
			default: /* confWhere is NOT valid  */
				MsgBox("Matrox PowerDesk - Invalid CONF_ADJ (where)","drawLayout() received an invalid screen adjacency.", 1);
			break;
		}
	}
	else if(scrn1 == 1 && scrn2 == 0) {
		switch(confWhere) {
			case CONF_ADJ_LEFTOF:
				gtk_widget_set_uposition(wdgButResolution1, RIGHT_X, RIGHT_Y);
				gtk_widget_set_uposition(wdgButAdjust1, RIGHT_X, RIGHT_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, LEFT_X, LEFT_Y);
				gtk_widget_set_uposition(wdgButAdjust2,	LEFT_X,	LEFT_Y + wdgButResolution2->allocation.height);
			break;
			
			case CONF_ADJ_RIGHTOF:
				gtk_widget_set_uposition(wdgButResolution1, LEFT_X, LEFT_Y);
				gtk_widget_set_uposition(wdgButAdjust1, LEFT_X, LEFT_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, RIGHT_X, RIGHT_Y);
				gtk_widget_set_uposition(wdgButAdjust2,	RIGHT_X, RIGHT_Y + wdgButResolution2->allocation.height);
			break;
			
			case CONF_ADJ_ABOVE:
				gtk_widget_set_uposition(wdgButResolution1, BELOW_X, BELOW_Y);
				gtk_widget_set_uposition(wdgButAdjust1, BELOW_X, BELOW_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, ABOVE_X, ABOVE_Y);
				gtk_widget_set_uposition(wdgButAdjust2,	ABOVE_X, ABOVE_Y + wdgButResolution2->allocation.height);
			break;
			
			case CONF_ADJ_BELOW:
				gtk_widget_set_uposition(wdgButResolution1, ABOVE_X, ABOVE_Y);
				gtk_widget_set_uposition(wdgButAdjust1, ABOVE_X, ABOVE_Y + wdgButResolution1->allocation.height);
				gtk_widget_set_uposition(wdgButResolution2, BELOW_X, BELOW_Y);
				gtk_widget_set_uposition(wdgButAdjust2,	BELOW_X, BELOW_Y + wdgButResolution2->allocation.height);
			break;
			
			default: /* confWhere is NOT valid */
				MsgBox(	"Matrox PowerDesk - Invalid CONF_ADJ (where)",
						"drawLayout() received an invalid screen adjacency.",
						1);
			break;
		}
	}
	else {/* we only support 2 screens so index should be 0 or 1 only  */
		MsgBox(	"Matrox PowerDesk - Invalid screen index",
				"drawLayout() received an invalid screen index.",
				1);
	}
}

/***********************************************
* set_window_on_top send a message to window manager
* requesting on-top position. This code is only
* valid for gnome and doesn't work with KDE.
* Taken from gtk-app-devel-list@gnome.org
* 
* IN=>	the_window : pointer to window widget.
* OUT=>	void
*************************************************/
static void set_window_on_top(GtkWidget* the_window)
{
	/* Gnome-only... to add : KDE. */
	
	/* Message that will be send to X */
	XClientMessageEvent xev;
	
	/* Fill the message struct */
	xev.type = ClientMessage;
	xev.window = GDK_WINDOW_XWINDOW(the_window->window);
	xev.message_type = XInternAtom(GDK_DISPLAY(), "_WIN_LAYER", False);
	xev.format = 32;
	xev.data.l[0] = 10;

	/* Send message to X */
	XSendEvent(GDK_DISPLAY(), GDK_ROOT_WINDOW(), False,
	SubstructureNotifyMask, (XEvent *) &xev);
}

/*********************************************************
* set_window_icon sets the window icon
* (I bet you guessed... hehe)
* 
* IN=>	the_window : pointer to window widget.
* OUT=>	void
**********************************************************/
static void set_window_icon(GtkWidget* this_window)
{
	#include "../pixmaps/icon.xpm" /* include xpm static array  */
	GdkPixmap *pixmap;
	GdkPixmap *mask;

	/* Create pixmap from xpm array */
	pixmap = gdk_pixmap_create_from_xpm_d(this_window->window, &mask,
					&this_window->style->bg[GTK_STATE_NORMAL], icon_xpm);

	gdk_window_set_icon(this_window->window, NULL, pixmap, mask);

	return;
}

/***********************************************************************
///////
//////  WinMsgBox window
/////
***********************************************************************/

int mustExit = 0;

/***********************************************************
* MsgBox implements a basic message box.
*
* IN=>	title : title of msg window
* 		msg :	message to display
* 		exit :	if 0, don't exit mgaXconf
* 				if 1, exit mgaXconf (bad error occured)
* OUT=>	void
***********************************************************/
void MsgBox(char* title, char* msg, int exit)
{
	GtkLabel*	lblMsg;
	
	/* Create window and set some proprieties */
	wdgWinMsgbox = create_winMsgBox();
	set_window_back(wdgWinMsgbox, lookup_widget(wdgWinMsgbox, "fixed8"));
	gtk_window_set_title(GTK_WINDOW(wdgWinMsgbox), title);
	set_window_on_top(wdgWinMsgbox);

	/* The message label */
	lblMsg = GTK_LABEL(lookup_widget(wdgWinMsgbox, "lblMsgBox"));
	gtk_label_set_text(lblMsg, msg);

	mustExit = exit;
	
	/* pop the window on screen */
	gtk_widget_show(wdgWinMsgbox);
}

void
on_winMsgBox_show(GtkWidget *widget, gpointer user_data)
{
	wdgWinMsgbox = widget;

    /* Set icon */
    set_window_icon(widget);
}


gboolean
on_winMsgBox_delete_event              (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    /* do not close */
	return TRUE;
}


void
on_buttonOKMsgBox_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	/* destroy the MsgBox window */
	gtk_widget_destroy(wdgWinMsgbox);
	wdgWinMsgbox = NULL;
	
	if(mustExit)
		gtk_main_quit();
	
}

/***********************************************************************
///////
//////  WinAdjust window
/////
***********************************************************************/

int itoa(char *string, int d)
{
    if(string) {
        return sprintf(string, "%d", d);
    }

    return -1;
}

int ftoa(char *string, float f)
{
    if(string) {
        return sprintf(string, "%1.3f", f);
    }
	
    return -1;
}

/*******************************************************************
* WriteVideoInfoOnTxtBox fills the textBox in Adjust window with
* valuable information about current modeline.
*
* IN=>	pMode : modeline to display informations about.
* OUT=>	void
*******************************************************************/
static void WriteVideoInfoOnTxtBox(XF86VidModeModeLine* pMode)
{

	char sValeur[256];
	int numberHeads;

	GtkText* txtParams = GTK_TEXT(lookup_widget(wdgWinAdjust,"txtParams"));

	/* Write informations about actual screens mode */	
	if(XineramaIsActive(pDisp))
		XineramaQueryScreens(pDisp, &numberHeads);
	else
		numberHeads = 1;
	
	if(numberHeads == 1)
	{
		char sInfo[255];
		XF86VidModeModeLine modeLine;
		int dotClock;

		XF86VidModeGetModeLine(pDisp, 0, &dotClock, &modeLine);
		
      snprintf(sInfo, 255, "Display 1: %d x %d, %d-bit, %d Hz",
							modeLine.hdisplay, 
							modeLine.vdisplay,
							DefaultDepth(pDisp,0),
	    					(int)((((dotClock * 1000) / (float)(modeLine.vtotal * modeLine.htotal)) * 100.0) + 50) / 100);
		
		gtk_label_set_text(GTK_LABEL(lookup_widget(wdgWinCentral, "lblResInfo")), sInfo);
	}
	else
	{
		char sInfo[255];
		XF86VidModeModeLine modeLine1, modeLine2;
		int dotClock1, dotClock2;

		XF86VidModeGetModeLine(pDisp, 0, &dotClock1, &modeLine1);
		XF86VidModeGetModeLine(pDisp, 1, &dotClock2, &modeLine2);
	
      snprintf(sInfo, 255, "Display 1: %d x %d, %d-bit, %d Hz\nDisplay 2: %d x %d, %d-bit, %d Hz",
							modeLine1.hdisplay, 
							modeLine1.vdisplay,
							DisplayPlanes(pDisp, 0),
	    					(int)((((dotClock1 * 1000) / (float)(modeLine1.vtotal * modeLine1.htotal)) * 100.0) + 50) / 100,
							modeLine2.hdisplay, 
							modeLine2.vdisplay,
							DisplayPlanes(pDisp, 0), /* Same here as X pretend that only 1 screen exist in Xinerama mode */
	    					(int)((((dotClock2 * 1000) / (float)(modeLine2.vtotal * modeLine2.htotal)) * 100.0) + 50) / 100);
		
		gtk_label_set_text(GTK_LABEL(lookup_widget(wdgWinCentral, "lblResInfo")), sInfo);
	}

	gtk_text_backward_delete(txtParams, gtk_text_get_length(txtParams));

	gtk_text_insert(txtParams, NULL, NULL, NULL,
               "Monitor settings:\n\n",19);

	gtk_text_insert(txtParams, NULL, NULL, NULL,
               "Horizontal resolution: ",23);
    itoa(sValeur, pMode->hdisplay);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Horizontal sync start: ",23);
    itoa(sValeur, pMode->hsyncstart);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Horizontal sync end: ",21);
    itoa(sValeur, pMode->hsyncend);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Horizontal total: ",18);
    itoa(sValeur, pMode->htotal);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n\n",2);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Vertical resolution: ",21);
    itoa(sValeur, pMode->vdisplay);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Vertical sync start: ",21);
    itoa(sValeur, pMode->vsyncstart);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Vertical sync end: ",19);
    itoa(sValeur, pMode->vsyncend);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);

    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "Vertical total: ",16);
    itoa(sValeur, pMode->vtotal);
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    sValeur, strlen(sValeur));
    gtk_text_insert(txtParams, NULL, NULL, NULL,
                    "\n",1);
}

/* Refresh every WinAdjust's widget that need to */
void ValuesCorrection(void)
{
    GtkLabel* lblPixClockVal;
    GtkSpinButton* sbuttonRefresh;
    char sValeur[255];
    int vrefresh;
    XF86VidModeMonitor monitor;
    int minRefresh = 0, maxRefresh = 0, i = 0;

	if(!wdgWinAdjust)
		return;

    XF86VidModeGetModeLine(pDisp, CurrentScreen, &device_info.dot_clock, &mode_line);
    XSync(pDisp, FALSE);

    sbuttonRefresh = GTK_SPIN_BUTTON(lookup_widget(wdgWinAdjust,"sbuttonRefresh"));
    lblPixClockVal = GTK_LABEL(lookup_widget(wdgWinAdjust,"lblPixelClockValue"));

    ftoa(sValeur, ((float) device_info.dot_clock) / 1000);
    gtk_label_set_text(lblPixClockVal, sValeur);

    /* Vertical Refresh calculation... also roundin' up the result. */
    vrefresh = ((((device_info.dot_clock * 1000) / (float)(mode_line.vtotal * mode_line.htotal)) * 100.0) + 50) / 100;

    XF86VidModeGetMonitor (pDisp, CurrentScreen, &monitor);

    /* Get highest and lowest possible value */
    minRefresh = monitor.vsync[0].lo;
    maxRefresh = monitor.vsync[0].hi;
    for(i = 0; i < monitor.nvsync; i++) {
		if(minRefresh > monitor.vsync[i].lo)
			minRefresh = monitor.vsync[i].lo;
		if(maxRefresh < monitor.vsync[i].hi)
			maxRefresh = monitor.vsync[i].hi;
	}

    gtk_spin_button_configure(	sbuttonRefresh, /* spinButton */
								GTK_ADJUSTMENT(gtk_adjustment_new(vrefresh,	minRefresh,	maxRefresh,1, 10, 10)),
    							1, /* climb rate */
    							0); /* digits */

	WriteVideoInfoOnTxtBox(&mode_line);

}

gboolean
on_winAdjust_delete_event                (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    if(bDidChange) {
       modeChange = modeChangeEXIT;
       wdgWinSave = create_winSaveMode();
       set_window_back(wdgWinSave, lookup_widget(wdgWinSave, "fixed2"));
       gtk_widget_show(wdgWinSave);
       return TRUE;
    }
	else {
       return FALSE;
    }
}

void
on_buttonAdjustDone_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bDidChange) {
       modeChange = modeChangeEXIT;
       wdgWinSave = create_winSaveMode();
       set_window_back(wdgWinSave, lookup_widget(wdgWinSave, "fixed2"));
       gtk_widget_show(wdgWinSave);
    }
    else {
		gtk_widget_destroy(wdgWinAdjust);
    }

}

void
on_winAdjust_show                        (GtkWidget       *widget,
                                        gpointer         user_data)
{
    wdgWinAdjust = widget;

    /* Set icon */
    set_window_icon(widget);
	
	/* Get correct ModeLine */
	XF86VidModeGetModeLine (pDisp, CurrentScreen, &device_info.dot_clock, &mode_line);

	/* Set values for TxtBox */
    ValuesCorrection();

	if((!bDriverModed) || (!bHalLoaded))
		MsgBox("Matrox PowerDesk","WARNING: Changes made are not supported.",0);
}


void
on_buttonLeft_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    moveScreenLeft(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonRight_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    moveScreenRight(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonDown_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    moveScreenDown(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonUp_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    moveScreenUp(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonWider_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    resizeScreenWider(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonNarrower_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    resizeScreenNarrower(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonShorter_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    resizeScreenShorter(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonTaller_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bHalLoaded){
    resizeScreenTaller(&mode_line, pDisp);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}


void
on_buttonPrev_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bDidChange) {
        modeChange = modeChangePREVIOUS;
        wdgWinSave = create_winSaveMode();
        set_window_back(wdgWinSave, lookup_widget(wdgWinSave, "fixed2"));
        gtk_widget_show(wdgWinSave);
    }
    else {
        if(bHalLoaded)
	{
	UndoLastChange(&mode_line, pDisp);
        }
	XF86VidModeLockModeSwitch(pDisp, CurrentScreen, FALSE);
        XF86VidModeSwitchMode(pDisp, CurrentScreen, -1);
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, TRUE);
        ValuesCorrection();
    }
}


void
on_buttonNext_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
    if(bDidChange) {
        modeChange = modeChangeNEXT;
        wdgWinSave = create_winSaveMode();
        set_window_back(wdgWinSave, lookup_widget(wdgWinSave, "fixed2"));
        gtk_widget_show(wdgWinSave);
    }
    else {
        if(bHalLoaded)
	{
	UndoLastChange(&mode_line, pDisp);
        }
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, FALSE);
        XF86VidModeSwitchMode(pDisp, CurrentScreen, 1);
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, TRUE);
        ValuesCorrection();
    }
}

void
on_buttonApply_pressed                 (GtkButton       *button,
                                        gpointer         user_data)
{
    int readRefresh = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(lookup_widget(wdgWinAdjust,"sbuttonRefresh")));

    if(bHalLoaded){
    applyNewRefresh(&mode_line, pDisp, readRefresh);
    bDidChange = TRUE;
    }
    ValuesCorrection();
}

gboolean
on_winAdjust_key_press_event             (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
    if(event->keyval == GDK_Escape) {
        UndoLastChange(&mode_line, pDisp);
        bDidChange = FALSE;
        ValuesCorrection();
        return TRUE;
    }
	else {
	    return FALSE;
	}
}

/***********************************************************************
///////
//////  WinSave window
/////
***********************************************************************/

void
on_winSaveMode_show                    (GtkWidget       *widget,
                                        gpointer         user_data)
{
    wdgWinSave = widget;
    set_window_icon(widget);
}

gboolean
on_winSaveMode_delete_event            (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	/* Make sure the window cannot be closed by the user... */
	return TRUE;
}


void
on_buttonSaveYes_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
    /*  Add the modeline */
    confAddModeline(pXF86Config, CurrentScreen, &mode_line, device_info.dot_clock);

    /* re-init CHANGE bool */
    bDidChange = FALSE;

    if(modeChange == modeChangeNEXT || modeChange == modeChangePREVIOUS) { /* If mode Switching */
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, FALSE);
        XF86VidModeSwitchMode(pDisp, CurrentScreen, modeChange);
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, TRUE);
        modeChange = modeChangeEXIT;
        ValuesCorrection();
    }
    else { /* If Exit */
		gtk_widget_destroy(wdgWinAdjust);
    }

    gtk_widget_destroy(wdgWinSave);
}


void
on_buttonSaveNo_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{

    /* Undo */
    UndoLastChange(&mode_line, pDisp);

    bDidChange = FALSE;

    if(modeChange == modeChangeNEXT || modeChange == modeChangePREVIOUS) /* If mode Switching */
    {
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, FALSE);
        XF86VidModeSwitchMode(pDisp, CurrentScreen, modeChange);
        XF86VidModeLockModeSwitch(pDisp, CurrentScreen, TRUE);
        modeChange = modeChangeEXIT;
        ValuesCorrection();
    }
    else /* If Exit */
    {
		gtk_widget_destroy(wdgWinAdjust);
    }

    gtk_widget_destroy(wdgWinSave);
}

/***********************************************************************
///////
//////  WinCentral window
/////
***********************************************************************/

void
on_buttonInvert_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{

	int scrn1, confWhere, scrn2;

    /* Rotate screens */
    if(where == CONF_ADJ_RIGHTOF)
    	where = CONF_ADJ_BELOW;
    else if(where == CONF_ADJ_LEFTOF)
    	where = CONF_ADJ_ABOVE;
    else if(where == CONF_ADJ_ABOVE)
    	where = CONF_ADJ_RIGHTOF;
    else if(where == CONF_ADJ_BELOW)
    	where = CONF_ADJ_LEFTOF;
    	
    /* Set Layout */
    confSetLayout(pXF86Config, where);

	confGetServerLayout(pXF86Config, &scrn1, &confWhere, &scrn2);
	drawLayout(scrn1, confWhere, scrn2);
}


void
on_buttonAdjust1_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	/* Popup the adjust window */
	wdgWinAdjust = create_winAdjust();
	CurrentScreen = 0;
	set_window_back(wdgWinAdjust, lookup_widget(wdgWinAdjust, "fixed1"));
	gtk_widget_show(wdgWinAdjust);
}


void
on_buttonAdjust2_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
	if(wasSingleHeadAtStartup)
	{
		/* need to restart ti adjust second screen if not available to driver */
		MsgBox(	"Matrox PowerDesk",
				"The secondary display of your DualHead hardware isn't currently active.\n\n"
				"To change settings for this secondary display, you need to first restart your operating system.",0);
	}
	else
	{
		/* Popup the adjust window */
		wdgWinAdjust = create_winAdjust();
		CurrentScreen = 1;
		set_window_back(wdgWinAdjust, lookup_widget(wdgWinAdjust, "fixed1"));
		gtk_widget_show(wdgWinAdjust);
	}
}

void
on_radioOff_toggled                    (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton))
	{
		where = 0;
		confSetLayout(pXF86Config, where);
	
		/* Just draw it LEFTOF but the Adjacency Layout is none (0) */
		drawLayout(0, CONF_ADJ_LEFTOF, 1);
	}
}


void
on_radioMultiDisplay_toggled           (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton))
	{
		int scrn1, scrn2, confWhere;
	
		/* Enable second screen buttons */
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonResolution2"), TRUE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonInvert"), TRUE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonAdjust2"), TRUE);

		if(where == 0)
			where = CONF_ADJ_LEFTOF;
		
		/* Enable DualHead */
		confEnableDualHead (pXF86Config, where);

		confGetServerLayout(pXF86Config, &scrn1, &confWhere, &scrn2);

		where = confWhere;

		drawLayout(scrn1, confWhere, scrn2);
	
		/* Enable automatic Xinerama */
		gtk_toggle_button_set_active(&(GTK_CHECK_BUTTON(lookup_widget(wdgWinCentral, "checkXinerama"))->toggle_button), TRUE);
	}
	else
	{
		/* Set DualHead Inactive */
		confDisableDualHead(pXF86Config);

		/* Disable automatic Xinerama */
		gtk_toggle_button_set_active(&(GTK_CHECK_BUTTON(lookup_widget(wdgWinCentral, "checkXinerama"))->toggle_button), FALSE);

		/* Disable second screen buttons */
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonResolution2"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonInvert"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonAdjust2"), FALSE);

	}
}


void
on_radioClone_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton))
	{
		/* Enable second screen buttons */
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonResolution2"), TRUE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonAdjust2"), TRUE);

		/* Set Clone Active */
    	confEnableClone(pXF86Config);
		
		where = 0;

		drawLayout(0, CONF_ADJ_LEFTOF, 1);
		
		/* Enable automatic Xinerama */
		gtk_toggle_button_set_active(&(GTK_CHECK_BUTTON(lookup_widget(wdgWinCentral, "checkXinerama"))->toggle_button), TRUE);

	}
	else
	{
		/* Set Clone Inactive */
		confDisableClone(pXF86Config);

		/* Disable automatic Xinerama */
		gtk_toggle_button_set_active(&(GTK_CHECK_BUTTON(lookup_widget(wdgWinCentral, "checkXinerama"))->toggle_button), FALSE);

		/* Disable second screen buttons */
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonResolution2"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonInvert"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonAdjust2"), FALSE);

	}
}

gboolean
on_winCentral_delete_event             (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    gtk_main_quit();
    return FALSE;
}

void
on_winCentral_show                     (GtkWidget       *widget,
                                        gpointer         user_data)
{
	GtkToggleButton* togButMultiDisplay = &(GTK_RADIO_BUTTON(lookup_widget(wdgWinCentral, "radioMultiDisplay"))->check_button).toggle_button;
	GtkToggleButton* togButClone =  &(GTK_RADIO_BUTTON(lookup_widget(wdgWinCentral, "radioClone"))->check_button).toggle_button;
	GtkCheckButton* checkHwCursor = GTK_CHECK_BUTTON(lookup_widget(wdgWinCentral, "checkHwCursor"));
	GtkCheckButton* checkXinerama = GTK_CHECK_BUTTON(lookup_widget(wdgWinCentral, "checkXinerama"));
	int numberHeads = 1;
	
	wdgWinCentral = widget;
    set_window_icon(widget);

	if (DEV_IS_G400() || DEV_IS_G450() || DEV_IS_G550())
	{
		/* Check for DualHead if active and place widget in appropriate state */
		if(confDualHeadIsActive(pXF86Config))
		{
			gtk_toggle_button_set_active(togButMultiDisplay, TRUE);
		}
		else if(confCloneIsActive(pXF86Config))
		{
			gtk_toggle_button_set_active(togButClone, TRUE);
			gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonInvert"), FALSE);
		}
		else
		{
			gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonResolution2"), FALSE);
			gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonAdjust2"), FALSE);
			gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonInvert"), FALSE);
		}
		
		if(XineramaIsActive(pDisp))
			XineramaQueryScreens(pDisp, &numberHeads);
		else
			numberHeads = 1;
	}
	else
	{
		numberHeads = 1;

		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "radioOff"), FALSE);
		gtk_widget_set_sensitive((GtkWidget *)togButMultiDisplay, FALSE);
		gtk_widget_set_sensitive((GtkWidget *)togButClone, FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonResolution2"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonAdjust2"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonInvert"), FALSE);
		gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "checkXinerama"), FALSE);
	}

	/* Check boxes states */
	gtk_toggle_button_set_active(&(checkXinerama->toggle_button), confXineramaIsActive(pXF86Config));
	gtk_toggle_button_set_active(&(checkHwCursor->toggle_button), confHWCursorIsActive(pXF86Config));
	
	if(confXineramaIsActive(pXF86Config))
		confUnloadModule(pXF86Config, "dri");
	
	if(numberHeads == 1)
	{
		char sInfo[255];
		XF86VidModeModeLine modeLine;
		int dotClock;

		XF86VidModeGetModeLine(pDisp, 0, &dotClock, &modeLine);
		
      snprintf(sInfo, 255, "Display 1: %d x %d, %d-bit, %d Hz",
							modeLine.hdisplay, 
							modeLine.vdisplay,
							DisplayPlanes(pDisp, 0),
	    					(int)((((dotClock * 1000) / (float)(modeLine.vtotal * modeLine.htotal)) * 100.0) + 50) / 100);
		
		gtk_label_set_text(GTK_LABEL(lookup_widget(wdgWinCentral, "lblResInfo")), sInfo);
	}
	else
	{
		char sInfo[255];
		XF86VidModeModeLine modeLine1, modeLine2;
		int dotClock1, dotClock2;

		XF86VidModeGetModeLine(pDisp, 0, &dotClock1, &modeLine1);
		XF86VidModeGetModeLine(pDisp, 1, &dotClock2, &modeLine2);
	
      snprintf(sInfo, 255, "Display 1: %d x %d, %d-bit, %d Hz\nDisplay 2: %d x %d, %d-bit, %d Hz",
							modeLine1.hdisplay, 
							modeLine1.vdisplay,
							DisplayPlanes(pDisp, 0),
	    					(int)((((dotClock1 * 1000) / (float)(modeLine1.vtotal * modeLine1.htotal)) * 100.0) + 50) / 100,
							modeLine2.hdisplay, 
							modeLine2.vdisplay,
							DisplayPlanes(pDisp,0),
	    					(int)((((dotClock2 * 1000) / (float)(modeLine2.vtotal * modeLine2.htotal)) * 100.0) + 50) / 100);
		
		gtk_label_set_text(GTK_LABEL(lookup_widget(wdgWinCentral, "lblResInfo")), sInfo);
	}

	if (!DEV_IS_G100() && !DEV_IS_G200() && !DEV_IS_G400() && !DEV_IS_G450() && !DEV_IS_G550())
	{
		bDriverModed = FALSE;
		MsgBox("Matrox PowerDesk",
			   "WARNING: Your graphics hardware is not compatible with this version of"
			   " Matrox PowerDesk. Some features are not supported.",
			   0);
	}
	else if(isDriverModed(&mode_line, pDisp))
	{
		bDriverModed = TRUE;

		if(isHalLoaded(&mode_line, pDisp))
			bHalLoaded = TRUE;
		else
		{
			bHalLoaded = FALSE;
			MsgBox(	"Matrox PowerDesk",
               		"WARNING: The Matrox HAL library (the \"mga_hal_drv.o\" file)"
					" is not detected. Some features are not supported.\n\nThe latest Matrox software is available at:\n"
					"http://www.matrox.com/mga",
					0);
		}

	}
	else
	{
		bDriverModed = FALSE;
      	MsgBox("Matrox PowerDesk",
			 "WARNING: The current display driver is not compatible with this version of Matrox PowerDesk."
			 " Some features are not supported.\n\nThe latest Matrox software is available at:\nhttp://www.matrox.com/mga",
			 0);
	}

}

void
on_checkXinerama_toggled               (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton) && (!(confXineramaIsActive(pXF86Config))))
   		confEnableXinerama(pXF86Config);
    else if(!gtk_toggle_button_get_active(togglebutton) && (confXineramaIsActive(pXF86Config)))
   		confDisableXinerama(pXF86Config);
}

void
on_checkHwCursor_toggled               (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{

	if(gtk_toggle_button_get_active(togglebutton) && (!(confHWCursorIsActive(pXF86Config))))
    	confEnableHWCursor(pXF86Config);
    else if(!gtk_toggle_button_get_active(togglebutton) && (confHWCursorIsActive(pXF86Config)))
    	confDisableHWCursor(pXF86Config);

}

void
on_buttonResolution1_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	
	CurrentScreen = 0;
	wdgWinResolutions = create_winResolutions();
	set_window_back(wdgWinResolutions, lookup_widget(wdgWinResolutions, "fixed6"));
	gtk_widget_show(wdgWinResolutions);

}


void
on_buttonResolution2_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	CurrentScreen = 1;
	wdgWinResolutions = create_winResolutions();
	set_window_back(wdgWinResolutions, lookup_widget(wdgWinResolutions, "fixed6"));
	gtk_widget_show(wdgWinResolutions);
}

void set_window_back_identify(GtkWidget* this_window, GtkWidget* fixed)
{
    #include "../pixmaps/screen.xpm"

    GtkStyle *style;
    GdkBitmap *mask;

    gtk_widget_realize(this_window);

    style = gtk_style_copy(gtk_widget_get_style(this_window));

    style->bg_pixmap[GTK_STATE_NORMAL] =
               gdk_pixmap_create_from_xpm_d(this_window->window, &mask,
                  &style->bg[GTK_STATE_NORMAL], screen_xpm);

    gtk_widget_set_style(this_window, style);
    gtk_widget_set_style(fixed, style);

    gtk_style_unref(style);

    return;
}

gboolean timeout_callback_destroy(gpointer NbHeads)
{
	int numberHeads = GPOINTER_TO_INT(NbHeads);
	int i = 0;
	
	/* Enable Identify button back... */
	gtk_widget_set_sensitive(lookup_widget(wdgWinCentral, "buttonIdentify"), TRUE);
	
	for(i = 0; i < numberHeads; i++)
	{
    	/* Destroy the identify windows. */
	    gtk_widget_destroy(wdgWinIdentify[i]);
	}

    return FALSE;
}

void
on_buttonIdentify_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
    int numberHeads = 1;
    int posX = 0, posY = 0;

    int i = 0;
    XineramaScreenInfo* xineramaInfo;

    char pixName[255];

    if(XineramaIsActive(pDisp))
	{
		/* Get Xinerama information */
		xineramaInfo = XineramaQueryScreens(pDisp, &numberHeads);
		
		/* Ignore screens not supported */
		if(numberHeads > MAXSCREENS) numberHeads = MAXSCREENS;
		
	 	for(i = 0; i < numberHeads; i++)
	 	{
			/* Get window pointer to clear it later... */
			wdgWinIdentify[i] = create_winIdentify();
		
	    	/* Set background */
    		set_window_back_identify(wdgWinIdentify[i], lookup_widget(wdgWinIdentify[i], "fixed4"));
    	
	    	/* Set window on top */
	    	gtk_window_set_transient_for(GTK_WINDOW(wdgWinIdentify[i]), GTK_WINDOW(wdgWinCentral));
	    	set_window_on_top(wdgWinIdentify[i]);
    	
	    	/* Set window at correct position and characteristics the window */
	    	sprintf(pixName, "pixS%d", i + 1);
	    	gtk_widget_show(lookup_widget(wdgWinIdentify[i], pixName));
			posX = (xineramaInfo + i)->x_org +
    			        ((xineramaInfo + i)->width / 2) - (wdgWinIdentify[0]->allocation.width / 2);
    		posY = (xineramaInfo + i)->y_org +
						((xineramaInfo + i)->height / 2) - (wdgWinIdentify[0]->allocation.height / 2);
		    gtk_widget_set_uposition(wdgWinIdentify[i], posX, posY);
	 	}
	 	
	 	/* Show all windows at same time... */
	 	for(i = 0; i < numberHeads; i++) gtk_widget_show(wdgWinIdentify[i]);

	}
	else /* Xinerama is NOT active */
	{
		/* One head only */
		numberHeads = 1;
		
		/* Get window pointer to clear it later... */
		wdgWinIdentify[0] = create_winIdentify();
		
    	/* Set background */
    	set_window_back_identify(wdgWinIdentify[0], lookup_widget(wdgWinIdentify[0], "fixed4"));
    	
    	/* Set window on top */
    	gtk_window_set_transient_for(GTK_WINDOW(wdgWinIdentify[0]), GTK_WINDOW(wdgWinCentral));
    	set_window_on_top(wdgWinIdentify[0]);
    	
    	/* display the window */
    	gtk_widget_show(lookup_widget(wdgWinIdentify[0], "pixS1"));
    	posX = (DisplayWidth(pDisp, 0) / 2) - (wdgWinIdentify[0]->allocation.width / 2);
    	posY = (DisplayHeight(pDisp, 0) / 2) - (wdgWinIdentify[0]->allocation.height / 2);
	    gtk_widget_set_uposition(wdgWinIdentify[0], posX, posY);
	    gtk_widget_show(wdgWinIdentify[0]);
	}

    /* Disable the button until it's finished... will set back in timeout_callback.*/
    gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
			
	/* Set the timeout callback on... see you in 2 or 3 seconds :) */
	gtk_timeout_add(IDENTIFY_TIME, timeout_callback_destroy, GINT_TO_POINTER(numberHeads));
}

void
on_buttonCentralOk_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
    /* Backup the Config file */
	rename(sConfigFile, sConfigBackup);
	
	/* Write config. file */
	if(!confWriteConfigFile(sConfigFile, pXF86Config))
	{
		/* If writing was not correctly done... */
		rename(sConfigBackup, sConfigFile);
		
    	MsgBox("Matrox PowerDesk",
			   "ERROR: The configuration file couldn't be written to. Settings haven't been changed.", 1);
		
	}
    else
    {
    	MsgBox("Matrox PowerDesk",
      "Changes have been saved to your configuration file. The previous file has been backed up.\n\nFor changes to take effect, you must restart your X server.", 1);
    }

    gtk_widget_destroy(wdgWinCentral);
}


void
on_buttonCentralCancel_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
    gtk_widget_destroy(wdgWinCentral);
    gtk_main_quit();
}

void
on_buttonCentralPreview_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{

	if(!wdgWinPreview)
	{
		/* Popup the preview window */
		wdgWinPreview = create_winPreview();
		set_window_back(wdgWinPreview, lookup_widget(wdgWinPreview, "fixed9"));
		gtk_widget_show(wdgWinPreview);
		gtk_label_set_text( GTK_LABEL(button->child), "Refresh Preview");
	}
	else
	{ 	/* update text box with new changes */
	    GtkRange* scrollRange;
	    GtkText* txtConfig = GTK_TEXT(lookup_widget(wdgWinPreview, "txtConfig"));
	    GtkVScrollbar* scrollConfig = GTK_VSCROLLBAR(lookup_widget(wdgWinPreview, "scrollConfig"));
	    FILE* tempFile;
	    char readLine[300];
	
	    /* Delete current text */
	    gtk_text_backward_delete(txtConfig, gtk_text_get_length(txtConfig));

	    /* freeze text box */
		gtk_text_freeze(txtConfig);
	
		/* Write struct to temp file */
		xf86writeConfigFile("/tmp/.mgaTempForXF86ConfFile", pXF86Config);
	
		tempFile = fopen("/tmp/.mgaTempForXF86ConfFile", "rt");
	
		/* Read file to widget */
		while(fgets(readLine, 300, tempFile))
	    	gtk_text_insert(txtConfig, NULL, NULL, NULL, readLine, -1);
	
	    /* unfreeze text box */
		gtk_text_thaw(txtConfig);
		
		/* delete temp file */
		remove("/tmp/.mgaTempForXF86ConfFile");
	
		/* assign scroll bar to text box */
		scrollRange = &(scrollConfig->scrollbar.range);
		gtk_range_set_adjustment(scrollRange, txtConfig->vadj);
		gtk_widget_hide(GTK_WIDGET(scrollConfig));  /* force redraw */
		gtk_widget_show(GTK_WIDGET(scrollConfig));
	
	}

}

void
on_buttonInfo_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	/* Popup the information window */
	wdgWinInfo = create_winInfo();
	set_window_back(wdgWinInfo, lookup_widget(wdgWinInfo, "fixed10"));
	gtk_widget_show(wdgWinInfo);

}

/***********************************************************************
///////
//////  WinOpen window
/////
***********************************************************************/

void
on_winOpen_show                        (GtkWidget       *widget,
                                        gpointer         user_data)
{
    const char *filename;
    char file[255];

    GtkEntry* entryOpen = GTK_ENTRY(lookup_widget(widget,"entryConfig"));
    GtkEntry* entryBackup = GTK_ENTRY(lookup_widget(widget,"entryBackup"));

    wdgWinOpen = widget;

    /* Set icon */
    set_window_icon(widget);

    /* Get default filename */
    filename = xf86openConfigFile(NULL ,NULL, NULL);

    if (filename)
    {
    	/* for working on a non-const string */
    	strcpy(file, filename);
    	
    	/* A default XF86Config was found... */
	    /* Set default text */
	    gtk_entry_set_text(entryOpen, file);
	    gtk_entry_set_text(entryBackup, strcat(file, "_MGA-BACKUP"));
	
	    xf86closeConfigFile();
    }
    else /* default not found */
	{
	    /* Set default text */
	    gtk_entry_set_text(entryOpen, "Please specify a configuration file.");
	    gtk_entry_set_text(entryBackup, "Please specify a backup file.");
	}

}

gboolean
on_winOpen_delete_event                (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{

    gtk_main_quit();
	return FALSE;
}

void
on_buttonOkOpenConfig_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{

    GtkEntry* entryOpen = GTK_ENTRY(lookup_widget(wdgWinOpen,"entryConfig"));
    GtkEntry* entryBackup = GTK_ENTRY(lookup_widget(wdgWinOpen,"entryBackup"));
	
	/* Init. global variables : filenames */
	strcpy(sConfigFile,gtk_entry_get_text(entryOpen));
	strcpy(sConfigBackup,gtk_entry_get_text(entryBackup));

	/* initialize XF86ConfigPtr pXF86Config with correct filename */
	pXF86Config = getConfigPtr(sConfigFile);
	
	/* if not a valid config file... */
	if(!pXF86Config)
	{
		gtk_widget_destroy(wdgWinOpen);
		MsgBox("Matrox PowerDesk",
			   "ERROR: The specified file isn't valid. Please make sure the file name refers to a valid configuration file.", 
			   1);
	}
	else /* opened with success */
	{
		int busNum, deviceNum;
		char busID[100];
		XF86ConfDevicePtr pConfDevice = pXF86Config->conf_device_lst;
		
		/* Check for a matrox device */
		if (!getMatroxPci(&deviceNum, &busNum, &device_info.id, &device_info.rev))
		{
			gtk_widget_destroy(wdgWinOpen);
			MsgBox("Matrox PowerDesk",
				   "ERROR: Your graphics hardware is not supported by this version of Matrox"
				   " PowerDesk.", 1);
		}
		else
		{
			/* Pop the Central window and destroy current window */
			wdgWinCentral = create_winCentral();
			gtk_widget_destroy(wdgWinOpen);
			set_window_back(wdgWinCentral, lookup_widget(wdgWinCentral, "fixed3"));
			gtk_widget_show(wdgWinCentral);

			/* Assign busID with cards */
			snprintf(busID, 100, "PCI:%d:%d:0", busNum, deviceNum);
			for(pConfDevice = pXF86Config->conf_device_lst; pConfDevice; pConfDevice = pConfDevice->list.next)
			{
				if(!strcmp(pConfDevice->dev_driver, "mga"))
				{
					free(pConfDevice->dev_busid);
					pConfDevice->dev_busid = calloc(strlen(busID) + 1, sizeof(char));
					strcpy(pConfDevice->dev_busid, busID);
				}
			}

			wasSingleHeadAtStartup = !(confDualHeadIsActive(pXF86Config) || confCloneIsActive(pXF86Config));
		}
	}
}


void
on_buttonCancelOpenConfig_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
    gtk_widget_destroy(wdgWinOpen);
    gtk_main_quit();
}

/***********************************************************************
///////
//////  WinResolutions window
/////
***********************************************************************/

int nbElementSelected = 0;

void
on_winResolutions_show                 (GtkWidget       *widget,
                                        gpointer         user_data)
{
	
	/* not more than 200 resolutions in the list.. it's huge anyway :) */
	gchar*			list_items[200] =	{
										"640x480",
										"800x600",
										"1024x768",
										"1152x864",
										"1280x1024",
										"1600x1200",
										};
	gchar*			selected_list_items[200] ;


	guint           nlist_items = 6;
	guint		nselected_list_items = 0;
	GList*		ResChildren;

	guint           i;
    GtkList*		listResolutions;
    GtkWidget*		list_item;
	
	GtkWidget*		radioYCComposite;
	GtkWidget*		radioSCARTComposite;
	GtkWidget*		radioSCARTRgb;
	GtkWidget*		radioSCARTType2;
	GtkWidget*		radioNormal;
	GtkWidget*		radioDigital;
	GtkWidget*		radioNTSC;
	GtkWidget*		radioPAL;
	GtkWidget*		radio8bpp;
	GtkWidget*		radio16bpp;
	GtkWidget*		radio24bpp;
	GtkWidget*		radio32bpp;
	
    XF86ConfModeLinePtr pConfModeLines;
    char			readRes[50];
	char*			cableType;
	char*			tvStandard;
	
	nbElementSelected = 0;
    wdgWinResolutions = widget;

    /* Set icon */
    set_window_icon(widget);

	/* Init widgets*/
	listResolutions = GTK_LIST(lookup_widget(wdgWinResolutions,"listResolutions"));
	radioYCComposite = lookup_widget(wdgWinResolutions,"radioYCComposite");
	radioSCARTComposite = lookup_widget(wdgWinResolutions,"radioSCARTComposite");
	radioSCARTRgb = lookup_widget(wdgWinResolutions,"radioSCARTRgb");
	radioSCARTType2 = lookup_widget(wdgWinResolutions,"radioSCARTType2");
	radioNormal = lookup_widget(wdgWinResolutions,"radioNormal");
	radioDigital = lookup_widget(wdgWinResolutions,"radioDigital");
	radioNTSC = lookup_widget(wdgWinResolutions,"radioNTSC");
	radioPAL = lookup_widget(wdgWinResolutions,"radioPAL");
	radio8bpp = lookup_widget(wdgWinResolutions,"radio8bpp");
	radio16bpp = lookup_widget(wdgWinResolutions,"radio16bpp");
	radio24bpp = lookup_widget(wdgWinResolutions,"radio24bpp");
	radio32bpp = lookup_widget(wdgWinResolutions,"radio32bpp");
	
	/* Cannot click OK if no res. choosen */
	gtk_widget_set_sensitive(lookup_widget(wdgWinResolutions,"buttonOkResolution"),FALSE);
	
	/* Get Modelines */
	pConfModeLines = confGetUserModeLines(pXF86Config, CurrentScreen);
	
	
	while(pConfModeLines)
	{
		snprintf(readRes, 50, pConfModeLines->ml_identifier);

		list_items[nlist_items] = calloc(strlen(readRes) + 1, sizeof(gchar)); /* one more for \0 */
		
		if(list_items[nlist_items] == NULL)
			MsgBox("Matrox PowerDesk","An internal error has occured. Your system may be out of memory.", 1);
		
		strcpy(list_items[nlist_items], readRes);
		nlist_items++;
		if(nlist_items >= 255) break;
		pConfModeLines = pConfModeLines->list.next;
	}
	
	/* Fill the resolutions list */
	for (i = 0; i < nlist_items; i++)
	{
		char*	sData;
		list_item = gtk_list_item_new_with_label(list_items[i]);
		sData = calloc(strlen(list_items[i]) + 1, sizeof(gchar));
		strcpy(sData, list_items[i]);
		gtk_object_set_user_data(GTK_OBJECT(list_item), (gpointer)sData);
		gtk_container_add(GTK_CONTAINER(listResolutions), list_item);
		gtk_widget_show(list_item);
	}

	/*select resolutions in list */
	nselected_list_items = confGetResolutions(pXF86Config, CurrentScreen, 
		confGetDefaultDepth(pXF86Config, CurrentScreen), selected_list_items, 200);
	
	ResChildren = gtk_container_children(GTK_CONTAINER(listResolutions));
	
	while(ResChildren) {
		for(i = 0; i < nselected_list_items; i++) {
			char* sData;
			GtkListItem* li;

			li = ResChildren->data;
			sData = (char*) gtk_object_get_user_data(GTK_OBJECT(li));

			
			if( strcmp(selected_list_items[i],sData) == 0) {
				gtk_list_item_select(li);
        			/*activate OK button*/
				if(!nbElementSelected)
			                gtk_widget_set_sensitive(lookup_widget(wdgWinResolutions,"buttonOkResolution"),TRUE);

			        nbElementSelected++;
				}
			}
				
			
		ResChildren = ResChildren->next;
		};


	/* Clean Up */
	for(i = 6; i < nlist_items; i++) free(list_items[i]);
	
	/* set bit depth */
	switch( confGetDefaultDepth(pXF86Config, CurrentScreen) ) 
	{
		case 8: gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radio8bpp)->check_button).toggle_button,TRUE); break;
		case 16: gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radio16bpp)->check_button).toggle_button,TRUE); break;
                case 24: gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radio24bpp)->check_button).toggle_button,TRUE); break;
                case 32: gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radio32bpp)->check_button).toggle_button,TRUE); break;
	}

	if (DEV_IS_G400())
	{
		/* support for DVI and TVout only for G400 cards */
		if(confTVIsActive(pXF86Config, CurrentScreen))
		{
			tvStandard = confGetTVStandard(pXF86Config, CurrentScreen);

			if(tvStandard && !strcmp(tvStandard, "PAL"))
				gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions, "radioPAL"))->check_button).toggle_button, TRUE);
			else
				gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions, "radioNTSC"))->check_button).toggle_button, TRUE);

			gtk_widget_set_sensitive(radioYCComposite, TRUE);
			gtk_widget_set_sensitive(radioSCARTComposite, TRUE);
			gtk_widget_set_sensitive(radioSCARTType2, TRUE);
			gtk_widget_set_sensitive(radioSCARTRgb, TRUE);

			cableType = confGetCableType(pXF86Config, CurrentScreen);
		
			if(cableType)
			{
				if(!strcmp(cableType, "SCART_COMPOSITE"))
					gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radioSCARTComposite)->check_button).toggle_button, TRUE);
				else if(!strcmp(cableType, "SCART_TYPE2"))
					gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radioSCARTType2)->check_button).toggle_button, TRUE);
				else if(!strcmp(cableType, "SCART_RGB"))
					gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radioSCARTRgb)->check_button).toggle_button, TRUE);
				else
					gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radioYCComposite)->check_button).toggle_button, TRUE);
			}
			else
				gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(radioYCComposite)->check_button).toggle_button, TRUE);
		}
		else
		{
			gtk_widget_set_sensitive(radioYCComposite, FALSE);
			gtk_widget_set_sensitive(radioSCARTComposite, FALSE);
			gtk_widget_set_sensitive(radioSCARTType2, FALSE);
			gtk_widget_set_sensitive(radioSCARTRgb, FALSE);

			if(confDigitalScreenIsActive(pXF86Config, CurrentScreen))
				gtk_toggle_button_set_active(&(GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions, "radioDigital"))->check_button).toggle_button, TRUE);
		
		}
	}
	else /* other cards */
	{
			/* disabling DVI options */
  			gtk_widget_set_sensitive(radioNormal, FALSE);
	  		gtk_widget_set_sensitive(radioDigital, FALSE);
	 	 	gtk_widget_set_sensitive(radioNTSC, FALSE);
 	 		gtk_widget_set_sensitive(radioPAL, FALSE);
		
			/* disabling TVout options */
			gtk_widget_set_sensitive(radioYCComposite, FALSE);
			gtk_widget_set_sensitive(radioSCARTComposite, FALSE);
			gtk_widget_set_sensitive(radioSCARTType2, FALSE);
			gtk_widget_set_sensitive(radioSCARTRgb, FALSE);
	}		
}

void
on_listResolutions_unselect_child      (GtkList         *list,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
	nbElementSelected--;
	
	if(!nbElementSelected)
		gtk_widget_set_sensitive(lookup_widget(wdgWinResolutions,"buttonOkResolution"),FALSE);

}


void
on_listResolutions_select_child        (GtkList         *list,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
	if(!nbElementSelected)
		gtk_widget_set_sensitive(lookup_widget(wdgWinResolutions,"buttonOkResolution"),TRUE);
	
	nbElementSelected++;
}


gboolean
on_winResolutions_delete_event         (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
    return FALSE;
}

static int QSortCompareBigToSmall(const void* p1, const void* p2)
{
	char sRead1[50];
	char sRead2[50];
	int  hdisp1;	/* Sort perfomed with hdisplay */
	int  hdisp2;
	
	/* Token function is destructive so I work on a copy. */
	strcpy(sRead1, *((char**)p1));
	strcpy(sRead2, *((char**)p2));
	
	hdisp1 = atoi(strtok(sRead1,"x"));
	hdisp2 = atoi(strtok(sRead2,"x"));

	if(hdisp1 < hdisp2)
		return 1;
	if(hdisp1 > hdisp2)
		return -1;
	else
		return 0;

}

static int QSortCompareSmallToBig(const void* p1, const void* p2)
{
	char sRead1[50];
	char sRead2[50];
	int  hdisp1;	/* Sort perfomed with hdisplay */
	int  hdisp2;
	
	/* Token function is destructive so I work on a copy. */
	strcpy(sRead1, *((char**)p1));
	strcpy(sRead2, *((char**)p2));
	
	hdisp1 = atoi(strtok(sRead1,"x"));
	hdisp2 = atoi(strtok(sRead2,"x"));

	if(hdisp1 < hdisp2)
		return -1;
	if(hdisp1 > hdisp2)
		return 1;
	else
		return 0;

}
void
on_radioNTSC_toggled                   (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget* 
		radioYCComposite = lookup_widget(wdgWinResolutions,"radioYCComposite");
	
	GtkWidget* 
		radioSCARTComposite = lookup_widget(wdgWinResolutions,"radioSCARTComposite");

	GtkWidget* 
		radioSCARTRgb = lookup_widget(wdgWinResolutions,"radioSCARTRgb");

	GtkWidget* 
		radioSCARTType2 = lookup_widget(wdgWinResolutions,"radioSCARTType2");
	
	if(gtk_toggle_button_get_active(togglebutton))
	{
		gtk_widget_set_sensitive(radioYCComposite, TRUE);
		gtk_widget_set_sensitive(radioSCARTComposite, TRUE);
		gtk_widget_set_sensitive(radioSCARTRgb, TRUE);
		gtk_widget_set_sensitive(radioSCARTType2, TRUE);
	}
	
}


void
on_radioPAL_toggled                    (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget* 
		radioYCComposite = lookup_widget(wdgWinResolutions,"radioYCComposite");
	
	GtkWidget* 
		radioSCARTComposite = lookup_widget(wdgWinResolutions,"radioSCARTComposite");

	GtkWidget* 
		radioSCARTRgb = lookup_widget(wdgWinResolutions,"radioSCARTRgb");

	GtkWidget* 
		radioSCARTType2 = lookup_widget(wdgWinResolutions,"radioSCARTType2");

	if(gtk_toggle_button_get_active(togglebutton))
	{
		gtk_widget_set_sensitive(radioYCComposite, TRUE);
		gtk_widget_set_sensitive(radioSCARTComposite, TRUE);
		gtk_widget_set_sensitive(radioSCARTRgb, TRUE);
		gtk_widget_set_sensitive(radioSCARTType2, TRUE);
	}
	
}

void
on_radioNormal_toggled                 (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget* 
		radioYCComposite = lookup_widget(wdgWinResolutions,"radioYCComposite");
	
	GtkWidget* 
		radioSCARTComposite = lookup_widget(wdgWinResolutions,"radioSCARTComposite");

	GtkWidget* 
		radioSCARTRgb = lookup_widget(wdgWinResolutions,"radioSCARTRgb");

	GtkWidget* 
		radioSCARTType2 = lookup_widget(wdgWinResolutions,"radioSCARTType2");

	if(gtk_toggle_button_get_active(togglebutton))
	{
		gtk_widget_set_sensitive(radioYCComposite, FALSE);
		gtk_widget_set_sensitive(radioSCARTComposite, FALSE);
		gtk_widget_set_sensitive(radioSCARTRgb, FALSE);
		gtk_widget_set_sensitive(radioSCARTType2, FALSE);
	}
	

}

void
on_radioDigital_toggled                (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget* 
		radioYCComposite = lookup_widget(wdgWinResolutions,"radioYCComposite");
	
	GtkWidget* 
		radioSCARTComposite = lookup_widget(wdgWinResolutions,"radioSCARTComposite");

	GtkWidget* 
		radioSCARTRgb = lookup_widget(wdgWinResolutions,"radioSCARTRgb");

	GtkWidget* 
		radioSCARTType2 = lookup_widget(wdgWinResolutions,"radioSCARTType2");

	if(gtk_toggle_button_get_active(togglebutton))
	{
		gtk_widget_set_sensitive(radioYCComposite, FALSE);
		gtk_widget_set_sensitive(radioSCARTComposite, FALSE);
		gtk_widget_set_sensitive(radioSCARTRgb, FALSE);
		gtk_widget_set_sensitive(radioSCARTType2, FALSE);
	}

}

void
on_buttonOkResolution_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
    char* pResolution[200];
    int nbResolution = 0;
    int colorDepth = 16;
    int i;
    GList*	items;
    GtkList* listResolutions = GTK_LIST(lookup_widget(wdgWinResolutions,"listResolutions"));

    GtkRadioButton* radio8bpp = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radio8bpp"));
    GtkRadioButton* radio16bpp = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radio16bpp"));
    GtkRadioButton* radio24bpp = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radio24bpp"));
    GtkRadioButton* radio32bpp = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radio32bpp"));

	GtkRadioButton* radioNTSC = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioNTSC"));
	GtkRadioButton* radioPAL = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioPAL"));
	GtkRadioButton* radioNormal = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioNormal"));
	GtkRadioButton* radioDigital = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioDigital"));

	GtkRadioButton* radioSCARTComposite = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioSCARTComposite"));
	GtkRadioButton* radioSCARTRgb = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioSCARTRgb"));
	GtkRadioButton* radioSCARTType2 = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioSCARTType2"));

	GtkRadioButton* radioSmall2Big = GTK_RADIO_BUTTON(lookup_widget(wdgWinResolutions,"radioSmall2Big"));

    if( gtk_toggle_button_get_active(&((radio8bpp->check_button).toggle_button)) )
    	colorDepth = 8;
    else if( gtk_toggle_button_get_active(&((radio16bpp->check_button).toggle_button)) )
    	colorDepth = 16;
    else if( gtk_toggle_button_get_active(&((radio24bpp->check_button).toggle_button)) )
    	colorDepth = 24;
    else if( gtk_toggle_button_get_active(&((radio32bpp->check_button).toggle_button)) )
    	colorDepth = 32;
    	
	for(items=GTK_LIST(listResolutions)->selection; items; items = items->next)
	{
		if (GTK_IS_LIST_ITEM(items->data))
		{
			pResolution[nbResolution] = calloc(50, sizeof(char));
			strcpy(pResolution[nbResolution], (char*)gtk_object_get_user_data(items->data));
			free((char*)gtk_object_get_user_data(items->data));
			nbResolution++;
		}
	}
	
	/* Sort the resolution array */
	if(gtk_toggle_button_get_active(&(radioSmall2Big->check_button).toggle_button))
		qsort((void*)pResolution, nbResolution, sizeof(char**), QSortCompareSmallToBig);
	else
		qsort((void*)pResolution, nbResolution, sizeof(char**), QSortCompareBigToSmall);

	if( (confDualHeadIsActive(pXF86Config)) || (confCloneIsActive(pXF86Config)) ) {
		confSetDefaultDepth(pXF86Config, 0, colorDepth);
		confSetDefaultDepth(pXF86Config, 1, colorDepth);
		}
	else 
		confSetDefaultDepth(pXF86Config, CurrentScreen, colorDepth);

	/* Set resolutions, same res for all bitdepths */
	confSetResolutions(pXF86Config, CurrentScreen, nbResolution, pResolution, 8);
	confSetResolutions(pXF86Config, CurrentScreen, nbResolution, pResolution, 16);
	confSetResolutions(pXF86Config, CurrentScreen, nbResolution, pResolution, 24);
	confSetResolutions(pXF86Config, CurrentScreen, nbResolution, pResolution, 32);
	
	/* Cleaning up */
	for(i = 0; i < nbResolution; i++) free( pResolution[i] );

	/* TVout, Digital or Normal Screen */
	if(gtk_toggle_button_get_active(&(radioNormal->check_button).toggle_button))
	{
		confDisableTVout(pXF86Config, CurrentScreen);
		confDisableDigitalScreen(pXF86Config, CurrentScreen);
	}
	else if(gtk_toggle_button_get_active(&(radioDigital->check_button).toggle_button))
	{
		confDisableTVout(pXF86Config, CurrentScreen);
		confEnableDigitalScreen(pXF86Config, CurrentScreen);
	}
	else if(gtk_toggle_button_get_active(&(radioNTSC->check_button).toggle_button))
	{
		char cableType[100];
	
		confDisableDigitalScreen(pXF86Config, CurrentScreen);
		
		if(gtk_toggle_button_get_active(&(radioSCARTComposite->check_button).toggle_button))
			strcpy(cableType, "SCART_COMPOSITE");
		else if(gtk_toggle_button_get_active(&(radioSCARTType2->check_button).toggle_button))
			strcpy(cableType, "SCART_TYPE2");
		else if(gtk_toggle_button_get_active(&(radioSCARTRgb->check_button).toggle_button))
			strcpy(cableType, "SCART_RGB");
		else 
			strcpy(cableType, "YC_COMPOSITE");
		
		confEnableTVout(pXF86Config, CurrentScreen, "NTSC", cableType);
	}
	else if(gtk_toggle_button_get_active(&(radioPAL->check_button).toggle_button))
	{
		char cableType[100];
	
		confDisableDigitalScreen(pXF86Config, CurrentScreen);
		
		if(gtk_toggle_button_get_active(&(radioSCARTComposite->check_button).toggle_button))
			strcpy(cableType, "SCART_COMPOSITE");
		else if(gtk_toggle_button_get_active(&(radioSCARTType2->check_button).toggle_button))
			strcpy(cableType, "SCART_TYPE2");
		else if(gtk_toggle_button_get_active(&(radioSCARTRgb->check_button).toggle_button))
			strcpy(cableType, "SCART_RGB");
		else 
			strcpy(cableType, "YC_COMPOSITE");
		
		confEnableTVout(pXF86Config, CurrentScreen, "PAL", cableType);
	}
	
	gtk_widget_destroy(wdgWinResolutions);
}

void
on_buttonCancelResolution_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
    gtk_widget_destroy(wdgWinResolutions);
}

/***********************************************************************
///////
//////  WinPreview window
/////
***********************************************************************/

gboolean
on_winPreview_delete_event             (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	wdgWinPreview = NULL;
	gtk_label_set_text(GTK_LABEL(GTK_BUTTON(lookup_widget(wdgWinCentral, "buttonCentralPreview"))->child), "Preview Config. File");
	return FALSE;
}


void
on_winPreview_show                     (GtkWidget       *widget,
                                        gpointer         user_data)
{
    GtkRange* scrollRange;
    GtkText* txtConfig = GTK_TEXT(lookup_widget(wdgWinPreview, "txtConfig"));
    GtkVScrollbar* scrollConfig = GTK_VSCROLLBAR(lookup_widget(wdgWinPreview, "scrollConfig"));
    FILE* tempFile;
    char readLine[300];

    /* Set icon */
    set_window_icon(widget);

	wdgWinPreview = widget;
	
    /* Delete current text */
    gtk_text_backward_delete(txtConfig, gtk_text_get_length(txtConfig));
	
    /* freeze text box */
	gtk_text_freeze(txtConfig);

	/* Write struct to temp file */
    xf86writeConfigFile ("/tmp/.mgaTempForXF86ConfFile", pXF86Config);
	
	tempFile = fopen("/tmp/.mgaTempForXF86ConfFile", "rt");
	
	/* Read file to widget */
	while(fgets(readLine, 300, tempFile))
	    gtk_text_insert(txtConfig, NULL, NULL, NULL, readLine, -1);
	
    /* unfreeze text box */
	gtk_text_thaw(txtConfig);
	
	/* delete temp file */
	remove("/tmp/.mgaTempForXF86ConfFile");
	
	/* assign scroll bar to text box */
	scrollRange = &(scrollConfig->scrollbar.range);
	gtk_range_set_adjustment(scrollRange, txtConfig->vadj);
	gtk_widget_hide(GTK_WIDGET(scrollConfig));  /* force redraw */
	gtk_widget_show(GTK_WIDGET(scrollConfig));
	
}

void
on_buttonPreviewClose_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
    gtk_widget_destroy(wdgWinPreview);
    wdgWinPreview = NULL;
	gtk_label_set_text(GTK_LABEL(GTK_BUTTON(lookup_widget(wdgWinCentral, "buttonCentralPreview"))->child), "Preview Config. File");
}


/***********************************************************************
///////
//////  WinInfo window
/////
***********************************************************************/

gboolean
on_winInfo_delete_event                (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{

  return FALSE;
}

void
on_buttonInfoOk_clicked                (GtkButton       *button,
                                        gpointer         user_data)
{
    gtk_widget_destroy(wdgWinInfo);
}


void
on_winInfo_show                        (GtkWidget       *widget,
                                        gpointer         user_data)
{
    GtkLabel*		lblXinerama = GTK_LABEL(lookup_widget(wdgWinInfo, "lblXinerama"));
    GtkLabel*		lblBiosVersion = GTK_LABEL(lookup_widget(wdgWinInfo, "lblBiosVersion"));
	GtkLabel*		lblVersion = GTK_LABEL(lookup_widget(wdgWinInfo, "lblVersion"));
	GtkLabel*		lblHAL = GTK_LABEL(lookup_widget(wdgWinInfo, "lblHAL"));
	char sBuffer[50];
	char sMgaXconfVersion[200];
	
	wdgWinInfo = widget;
    set_window_icon(widget);

	/* lblXinerama state On/OFF */
	if(XineramaIsActive(pDisp))
		gtk_label_set_text(lblXinerama, "Xinerama: Enabled");
	else
		gtk_label_set_text(lblXinerama, "Xinerama: Disabled");

	/* Bios Version */
	DisplayBiosVersion(sBuffer, 50);
	if (sBuffer)
	{
		char sBiosVersion[150];
      snprintf(sBiosVersion, 150, "Graphics BIOS version: %s", sBuffer);
		gtk_label_set_text(lblBiosVersion, sBiosVersion);
	}
	else
	{
      gtk_label_set_text(lblBiosVersion, "Graphics BIOS version: --");
	}

	/* mgaXconf version */
   snprintf(sMgaXconfVersion, 200, "Matrox PowerDesk (Linux) version: %d.%02d.%d", MAJOR, MINOR, BUILD);
	gtk_label_set_text(lblVersion, sMgaXconfVersion);

	if(bDriverModed)
	{
		/* Hal loaded or not label */
		if(bHalLoaded)
         gtk_label_set_text(lblHAL, "Matrox HAL library: Installed");
		else
         gtk_label_set_text(lblHAL, "Matrox HAL library: Not installed");
	}

}

